// Funciones principales del sistema

// Inicializar tooltips
document.addEventListener('DOMContentLoaded', function() {
    // Inicializar tooltips de Bootstrap
    var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
    var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
        return new bootstrap.Tooltip(tooltipTriggerEl);
    });
    
    // Inicializar popovers
    var popoverTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="popover"]'));
    var popoverList = popoverTriggerList.map(function (popoverTriggerEl) {
        return new bootstrap.Popover(popoverTriggerEl);
    });
    
    // Mostrar mensajes de éxito automáticamente
    mostrarMensajesAutomaticos();
    
    // Configurar validación de formularios
    configurarValidacionFormularios();
});

// Función para mostrar mensajes emergentes
function mostrarMensaje(tipo, mensaje, tiempo = 3000) {
    // Crear elemento de alerta
    var alertDiv = document.createElement('div');
    alertDiv.className = 'alert alert-' + tipo + ' alert-dismissible fade show position-fixed';
    alertDiv.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px; box-shadow: 0 5px 15px rgba(0,0,0,0.1);';
    
    alertDiv.innerHTML = `
        <div class="d-flex align-items-center">
            <i class="fas ${obtenerIcono(tipo)} me-3 fs-4"></i>
            <div class="flex-grow-1">
                <strong>${obtenerTitulo(tipo)}</strong><br>
                <span>${mensaje}</span>
            </div>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    `;
    
    // Añadir al body
    document.body.appendChild(alertDiv);
    
    // Remover después del tiempo especificado
    setTimeout(function() {
        alertDiv.classList.remove('show');
        setTimeout(function() {
            if (alertDiv.parentNode) {
                alertDiv.parentNode.removeChild(alertDiv);
            }
        }, 300);
    }, tiempo);
    
    // Retornar el elemento para control manual
    return alertDiv;
}

// Obtener icono según tipo de mensaje
function obtenerIcono(tipo) {
    switch(tipo) {
        case 'success': return 'fa-check-circle';
        case 'danger': return 'fa-exclamation-circle';
        case 'warning': return 'fa-exclamation-triangle';
        case 'info': return 'fa-info-circle';
        default: return 'fa-bell';
    }
}

// Obtener título según tipo de mensaje
function obtenerTitulo(tipo) {
    switch(tipo) {
        case 'success': return '¡Éxito!';
        case 'danger': return '¡Error!';
        case 'warning': return '¡Advertencia!';
        case 'info': return 'Información';
        default: return 'Mensaje';
    }
}

// Mostrar mensajes automáticos de la sesión
function mostrarMensajesAutomaticos() {
    // Verificar si hay mensajes en sessionStorage
    var mensaje = sessionStorage.getItem('mensaje_sistema');
    var tipo = sessionStorage.getItem('tipo_mensaje');
    
    if (mensaje && tipo) {
        mostrarMensaje(tipo, mensaje);
        
        // Limpiar sessionStorage
        sessionStorage.removeItem('mensaje_sistema');
        sessionStorage.removeItem('tipo_mensaje');
    }
}

// Configurar mensaje para mostrar después de redirección
function setMensajeRedireccion(mensaje, tipo) {
    sessionStorage.setItem('mensaje_sistema', mensaje);
    sessionStorage.setItem('tipo_mensaje', tipo);
}

// Configurar validación de formularios
function configurarValidacionFormularios() {
    // Seleccionar todos los formularios que requieren validación
    var forms = document.querySelectorAll('.needs-validation');
    
    // Iterar sobre cada formulario
    Array.prototype.slice.call(forms).forEach(function(form) {
        form.addEventListener('submit', function(event) {
            if (!form.checkValidity()) {
                event.preventDefault();
                event.stopPropagation();
                
                // Mostrar mensaje de error
                mostrarMensaje('warning', 'Por favor complete todos los campos requeridos correctamente.');
            }
            
            form.classList.add('was-validated');
        }, false);
    });
}

// Consultar DNI mediante API
async function consultarDNI(dni) {
    if (dni.length !== 8) {
        throw new Error('El DNI debe tener 8 dígitos');
    }
    
    try {
        // URL de ejemplo - Reemplazar con API real
        const url = `https://api.apis.net.pe/v1/dni?numero=${dni}`;
        
        const response = await fetch(url, {
            method: 'GET',
            headers: {
                'Accept': 'application/json',
            }
        });
        
        if (!response.ok) {
            throw new Error('Error en la consulta');
        }
        
        const data = await response.json();
        return data;
        
    } catch (error) {
        console.error('Error consultando DNI:', error);
        throw error;
    }
}

// Formatear fecha
function formatearFecha(fecha, formato = 'dd/mm/yyyy') {
    const date = new Date(fecha);
    
    if (isNaN(date.getTime())) {
        return fecha;
    }
    
    const dia = String(date.getDate()).padStart(2, '0');
    const mes = String(date.getMonth() + 1).padStart(2, '0');
    const año = date.getFullYear();
    const horas = String(date.getHours()).padStart(2, '0');
    const minutos = String(date.getMinutes()).padStart(2, '0');
    
    switch(formato) {
        case 'dd/mm/yyyy':
            return `${dia}/${mes}/${año}`;
        case 'yyyy-mm-dd':
            return `${año}-${mes}-${dia}`;
        case 'dd/mm/yyyy HH:MM':
            return `${dia}/${mes}/${año} ${horas}:${minutos}`;
        default:
            return `${dia}/${mes}/${año}`;
    }
}

// Exportar tabla a Excel
function exportarExcel(tablaId, nombreArchivo = 'reporte.xlsx') {
    const tabla = document.getElementById(tablaId);
    if (!tabla) {
        mostrarMensaje('error', 'No se encontró la tabla para exportar');
        return;
    }
    
    // Crear libro de Excel (simplificado)
    let csv = [];
    const filas = tabla.querySelectorAll('tr');
    
    for (let fila of filas) {
        const columnas = fila.querySelectorAll('td, th');
        let filaCSV = [];
        
        for (let columna of columnas) {
            // Remover contenido de botones y acciones
            if (!columna.querySelector('button')) {
                filaCSV.push(`"${columna.innerText.replace(/"/g, '""')}"`);
            }
        }
        
        csv.push(filaCSV.join(','));
    }
    
    // Descargar archivo
    const csvString = csv.join('\n');
    const blob = new Blob([csvString], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    
    if (navigator.msSaveBlob) {
        // Para IE
        navigator.msSaveBlob(blob, nombreArchivo);
    } else {
        link.href = URL.createObjectURL(blob);
        link.download = nombreArchivo;
        link.style.display = 'none';
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
    }
    
    mostrarMensaje('success', 'Reporte exportado exitosamente');
}

// Cargar datos por AJAX
function cargarDatos(url, callback) {
    fetch(url)
        .then(response => response.json())
        .then(data => callback(data))
        .catch(error => {
            console.error('Error cargando datos:', error);
            mostrarMensaje('error', 'Error cargando datos');
        });
}

// Validar email
function validarEmail(email) {
    const regex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return regex.test(email);
}

// Mostrar/ocultar contraseña
function togglePassword(inputId) {
    const input = document.getElementById(inputId);
    const toggleBtn = document.querySelector(`[data-toggle="${inputId}"]`);
    
    if (input.type === 'password') {
        input.type = 'text';
        toggleBtn.innerHTML = '<i class="fas fa-eye-slash"></i>';
    } else {
        input.type = 'password';
        toggleBtn.innerHTML = '<i class="fas fa-eye"></i>';
    }
}

// Contador de caracteres
function contadorCaracteres(inputId, maxLength) {
    const input = document.getElementById(inputId);
    const contador = document.getElementById(`${inputId}-contador`);
    
    if (!input || !contador) return;
    
    input.addEventListener('input', function() {
        const longitud = this.value.length;
        contador.textContent = `${longitud}/${maxLength}`;
        
        if (longitud > maxLength) {
            contador.style.color = 'var(--danger)';
        } else if (longitud > maxLength * 0.8) {
            contador.style.color = 'var(--warning)';
        } else {
            contador.style.color = 'inherit';
        }
    });
}

// Formatear número con separadores
function formatearNumero(numero) {
    return numero.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ",");
}

// Confirmar acción
function confirmarAccion(mensaje, callback) {
    if (confirm(mensaje)) {
        callback();
    }
}

// Inicializar datepickers
function inicializarDatepickers() {
    const datepickers = document.querySelectorAll('.datepicker');
    
    datepickers.forEach(datepicker => {
        datepicker.type = 'date';
        datepicker.max = new Date().toISOString().split('T')[0];
    });
}

// Cargar select con opciones
function cargarSelect(selectId, datos, valorKey, textoKey, primeraOpcion = 'Seleccione una opción') {
    const select = document.getElementById(selectId);
    if (!select) return;
    
    // Limpiar opciones actuales
    select.innerHTML = '';
    
    // Agregar primera opción
    if (primeraOpcion) {
        const option = document.createElement('option');
        option.value = '';
        option.textContent = primeraOpcion;
        select.appendChild(option);
    }
    
    // Agregar opciones de datos
    datos.forEach(dato => {
        const option = document.createElement('option');
        option.value = dato[valorKey];
        option.textContent = dato[textoKey];
        select.appendChild(option);
    });
}