<?php
// Habilitar errores temporalmente
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

session_start();

// Verificar sesión y permisos de administrador
if (!isset($_SESSION['user_id'])) {
    header("Location: ../../login.php");
    exit();
}

// Verificar que sea administrador
if (!isset($_SESSION['user_tipo']) || $_SESSION['user_tipo'] != 'admin') {
    $_SESSION['mensaje'] = "No tienes permisos para acceder a esta sección";
    $_SESSION['tipo_mensaje'] = "danger";
    header("Location: ../../dashboard.php");
    exit();
}

require_once '../../config/database.php';

$database = new Database();
$db = $database->getConnection();

if (!$db) {
    die("Error de conexión a la base de datos");
}

// Obtener datos actuales de la entidad
$entidad = [];
try {
    $query = "SELECT * FROM entidad LIMIT 1";
    $stmt = $db->prepare($query);
    $stmt->execute();
    $entidad = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Si no existe, crear registro por defecto
    if (!$entidad) {
        $query = "INSERT INTO entidad (codigo, nombre_completo) VALUES ('ENT001', 'Entidad Pública del Estado')";
        $db->exec($query);
        
        // Obtener el registro creado
        $stmt = $db->prepare("SELECT * FROM entidad LIMIT 1");
        $stmt->execute();
        $entidad = $stmt->fetch(PDO::FETCH_ASSOC);
    }
} catch (PDOException $e) {
    error_log("Error al cargar datos de la entidad: " . $e->getMessage());
}

$mensaje = '';
$tipo_mensaje = '';

// Procesar actualización de datos
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $codigo = trim($_POST['codigo'] ?? '');
    $nombre_completo = trim($_POST['nombre_completo'] ?? '');
    $direccion = trim($_POST['direccion'] ?? '');
    $telefono = trim($_POST['telefono'] ?? '');
    $email = trim($_POST['email'] ?? '');
    
    // Validaciones
    if (empty($codigo) || empty($nombre_completo)) {
        $mensaje = "El código y nombre son obligatorios";
        $tipo_mensaje = "danger";
    } else {
        try {
            // Procesar logo si se subió
            $logo = $entidad['logo'] ?? '';
            
            if (isset($_FILES['logo']) && $_FILES['logo']['error'] == UPLOAD_ERR_OK) {
                $uploadDir = '../../assets/img/';
                
                // Crear directorio si no existe
                if (!is_dir($uploadDir)) {
                    mkdir($uploadDir, 0755, true);
                }
                
                $fileInfo = pathinfo($_FILES['logo']['name']);
                $extension = strtolower($fileInfo['extension'] ?? '');
                
                // Validar extensiones permitidas
                $allowedExtensions = ['jpg', 'jpeg', 'png', 'gif', 'svg'];
                
                if (in_array($extension, $allowedExtensions)) {
                    // Generar nombre único
                    $fileName = 'logo_' . time() . '.' . $extension;
                    $uploadFile = $uploadDir . $fileName;
                    
                    // Mover archivo
                    if (move_uploaded_file($_FILES['logo']['tmp_name'], $uploadFile)) {
                        // Eliminar logo anterior si existe
                        if ($logo && file_exists($uploadDir . $logo)) {
                            unlink($uploadDir . $logo);
                        }
                        $logo = $fileName;
                    } else {
                        $mensaje = "Error al subir el logo";
                        $tipo_mensaje = "warning";
                    }
                } else {
                    $mensaje = "Formato de archivo no permitido. Use: " . implode(', ', $allowedExtensions);
                    $tipo_mensaje = "warning";
                }
            }
            
            // Actualizar datos
            $query = "UPDATE entidad SET 
                      codigo = :codigo,
                      nombre_completo = :nombre_completo,
                      logo = :logo,
                      direccion = :direccion,
                      telefono = :telefono,
                      email = :email,
                      fecha_actualizacion = NOW()
                      WHERE id = :id";
            
            $stmt = $db->prepare($query);
            $stmt->bindParam(':codigo', $codigo);
            $stmt->bindParam(':nombre_completo', $nombre_completo);
            $stmt->bindParam(':logo', $logo);
            $stmt->bindParam(':direccion', $direccion);
            $stmt->bindParam(':telefono', $telefono);
            $stmt->bindParam(':email', $email);
            $stmt->bindParam(':id', $entidad['id'], PDO::PARAM_INT);
            
            if ($stmt->execute()) {
                $mensaje = "Configuración actualizada exitosamente";
                $tipo_mensaje = "success";
                
                // Actualizar datos en variable
                $entidad = array_merge($entidad, [
                    'codigo' => $codigo,
                    'nombre_completo' => $nombre_completo,
                    'logo' => $logo,
                    'direccion' => $direccion,
                    'telefono' => $telefono,
                    'email' => $email
                ]);
            } else {
                $mensaje = "Error al actualizar la configuración";
                $tipo_mensaje = "danger";
            }
            
        } catch (PDOException $e) {
            $mensaje = "Error: " . $e->getMessage();
            $tipo_mensaje = "danger";
            error_log("Error en actualización de entidad: " . $e->getMessage());
        }
    }
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Configuración - Sistema de Visitas</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        :root {
            --celeste-primario: #e3f2fd;
            --celeste-secundario: #bbdefb;
            --celeste-oscuro: #1976d2;
        }
        body {
            background-color: #f8f9fa;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        .main-container {
            margin-left: 250px;
            padding: 20px;
            min-height: 100vh;
        }
        .sidebar {
            background: linear-gradient(180deg, var(--celeste-oscuro) 0%, #1565c0 100%);
            min-height: 100vh;
            color: white;
            padding: 0;
            position: fixed;
            width: 250px;
        }
        .sidebar .nav-link {
            color: rgba(255,255,255,0.8);
            padding: 12px 20px;
            margin: 2px 10px;
            border-radius: 8px;
            transition: all 0.3s;
            text-decoration: none;
        }
        .sidebar .nav-link:hover, .sidebar .nav-link.active {
            background: rgba(255,255,255,0.1);
            color: white;
            transform: translateX(5px);
        }
        .card {
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.05);
            border: none;
            margin-bottom: 20px;
        }
        .card-header {
            background: linear-gradient(to right, #6f42c1, #9c27b0);
            color: white;
            border-radius: 15px 15px 0 0 !important;
            padding: 15px 20px;
            font-weight: 600;
        }
        .btn-primary {
            background: linear-gradient(to right, var(--celeste-oscuro), #2196f3);
            border: none;
            border-radius: 8px;
            padding: 10px 20px;
            transition: all 0.3s;
        }
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(33, 150, 243, 0.4);
        }
        .logo-preview {
            width: 150px;
            height: 150px;
            border: 2px dashed #dee2e6;
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            overflow: hidden;
            background: #f8f9fa;
            margin: 0 auto 20px;
            transition: all 0.3s;
        }
        .logo-preview:hover {
            border-color: var(--celeste-oscuro);
            transform: scale(1.05);
        }
        .logo-preview img {
            max-width: 100%;
            max-height: 100%;
            object-fit: contain;
        }
        .logo-placeholder {
            color: #6c757d;
            text-align: center;
            padding: 20px;
        }
        .form-control, .form-select {
            border-radius: 8px;
            border: 2px solid #e0e0e0;
            padding: 10px 15px;
            transition: all 0.3s;
        }
        .form-control:focus, .form-select:focus {
            border-color: var(--celeste-oscuro);
            box-shadow: 0 0 0 0.2rem rgba(25, 118, 210, 0.25);
        }
        .info-card {
            background: linear-gradient(135deg, #e3f2fd 0%, #bbdefb 100%);
            border-left: 5px solid var(--celeste-oscuro);
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
        }
        @media (max-width: 768px) {
            .sidebar {
                width: 100%;
                position: relative;
                min-height: auto;
            }
            .main-container {
                margin-left: 0;
            }
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 col-lg-2 d-md-block sidebar">
                <div class="position-sticky pt-3">
                    <div class="text-center mb-4 p-3">
                        <div class="p-3">
                            <i class="fas fa-building fa-3x mb-3"></i>
                            <h5>Sistema de Visitas</h5>
                            <small>Entidad Pública</small>
                        </div>
                    </div>
                    
                    <div class="nav flex-column">
                        <a class="nav-link" href="../../dashboard.php">
                            <i class="fas fa-home me-2"></i>Dashboard
                        </a>
                        <a class="nav-link" href="../visitas/registrar.php">
                            <i class="fas fa-sign-in-alt me-2"></i>Registrar Ingreso
                        </a>
                        <a class="nav-link" href="../visitas/salida.php">
                            <i class="fas fa-sign-out-alt me-2"></i>Registrar Salida
                        </a>
                        <a class="nav-link" href="../visitas/listar.php">
                            <i class="fas fa-list me-2"></i>Listar Visitas
                        </a>
                        <a class="nav-link" href="../reportes/generar.php">
                            <i class="fas fa-chart-bar me-2"></i>Reportes
                        </a>
                        
                        <hr class="my-3 text-white-50">
                        <h6 class="sidebar-heading px-3 mt-4 mb-1 text-white-50">
                            <span>Administración</span>
                        </h6>
                        <a class="nav-link" href="../usuarios/gestion.php">
                            <i class="fas fa-users me-2"></i>Gestión de Usuarios
                        </a>
                        <a class="nav-link" href="../areas/gestion.php">
                            <i class="fas fa-sitemap me-2"></i>Áreas
                        </a>
                        <a class="nav-link active" href="configuracion.php">
                            <i class="fas fa-cog me-2"></i>Configuración
                        </a>
                        
                        <hr class="my-3 text-white-50">
                        <a class="nav-link" href="../../logout.php">
                            <i class="fas fa-sign-out-alt me-2"></i>Cerrar Sesión
                        </a>
                    </div>
                </div>
            </div>

            <!-- Main Content -->
            <div class="col-md-9 ms-sm-auto col-lg-10 px-md-4 main-container">
                <!-- Header -->
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Configuración de la Entidad</h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <span class="badge bg-primary">
                            <i class="fas fa-user-shield me-1"></i>Administrador
                        </span>
                    </div>
                </div>
                
                <?php if ($mensaje): ?>
                <div class="alert alert-<?php echo $tipo_mensaje; ?> alert-dismissible fade show" role="alert">
                    <i class="fas fa-<?php echo $tipo_mensaje == 'success' ? 'check-circle' : 'exclamation-circle'; ?> me-2"></i>
                    <?php echo $mensaje; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
                <?php endif; ?>
                
                <!-- Información actual -->
                <div class="info-card">
                    <h5><i class="fas fa-info-circle me-2"></i>Información Actual</h5>
                    <div class="row">
                        <div class="col-md-6">
                            <p><strong>Última actualización:</strong> 
                                <?php echo $entidad['fecha_actualizacion'] ? date('d/m/Y H:i', strtotime($entidad['fecha_actualizacion'])) : 'No registrada'; ?>
                            </p>
                        </div>
                        <div class="col-md-6 text-end">
                            <span class="badge bg-info">
                                <i class="fas fa-code me-1"></i>Código: <?php echo htmlspecialchars($entidad['codigo'] ?? 'No definido'); ?>
                            </span>
                        </div>
                    </div>
                </div>
                
                <!-- Formulario de configuración -->
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="fas fa-cog me-2"></i>Editar Configuración</h5>
                    </div>
                    <div class="card-body">
                        <form method="POST" action="" enctype="multipart/form-data" id="formConfig">
                            <div class="row">
                                <div class="col-md-6">
                                    <!-- Logo -->
                                    <div class="logo-preview mb-4" id="logoPreview">
                                        <?php if (!empty($entidad['logo']) && file_exists('../../assets/img/' . $entidad['logo'])): ?>
                                            <img src="../../assets/img/<?php echo htmlspecialchars($entidad['logo']); ?>" 
                                                 alt="Logo actual" id="currentLogo">
                                        <?php else: ?>
                                            <div class="logo-placeholder">
                                                <i class="fas fa-building fa-3x mb-2"></i>
                                                <p class="mb-0">Logo no definido</p>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="logo" class="form-label">Logo de la Entidad</label>
                                        <input type="file" class="form-control" id="logo" name="logo" 
                                               accept=".jpg,.jpeg,.png,.gif,.svg">
                                        <small class="text-muted">Formatos permitidos: JPG, PNG, GIF, SVG. Tamaño máximo: 2MB</small>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="codigo" class="form-label">Código de la Entidad *</label>
                                        <input type="text" class="form-control" id="codigo" name="codigo" 
                                               value="<?php echo htmlspecialchars($entidad['codigo'] ?? ''); ?>" 
                                               required placeholder="Ej: ENT001">
                                    </div>
                                </div>
                                
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="nombre_completo" class="form-label">Nombre Completo *</label>
                                        <input type="text" class="form-control" id="nombre_completo" name="nombre_completo" 
                                               value="<?php echo htmlspecialchars($entidad['nombre_completo'] ?? ''); ?>" 
                                               required placeholder="Nombre completo de la entidad">
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="direccion" class="form-label">Dirección</label>
                                        <textarea class="form-control" id="direccion" name="direccion" 
                                                  rows="2" placeholder="Dirección de la entidad"><?php echo htmlspecialchars($entidad['direccion'] ?? ''); ?></textarea>
                                    </div>
                                    
                                    <div class="row">
                                        <div class="col-md-6 mb-3">
                                            <label for="telefono" class="form-label">Teléfono</label>
                                            <input type="tel" class="form-control" id="telefono" name="telefono" 
                                                   value="<?php echo htmlspecialchars($entidad['telefono'] ?? ''); ?>" 
                                                   placeholder="Ej: +51 123456789">
                                        </div>
                                        
                                        <div class="col-md-6 mb-3">
                                            <label for="email" class="form-label">Correo Electrónico</label>
                                            <input type="email" class="form-control" id="email" name="email" 
                                                   value="<?php echo htmlspecialchars($entidad['email'] ?? ''); ?>" 
                                                   placeholder="contacto@entidad.gob.pe">
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <hr class="my-4">
                            
                            <div class="d-flex justify-content-between">
                                <a href="../../dashboard.php" class="btn btn-secondary">
                                    <i class="fas fa-arrow-left me-1"></i> Volver al Dashboard
                                </a>
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-save me-1"></i> Guardar Cambios
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
                
                <!-- Información del sistema -->
                <div class="row mt-4">
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header">
                                <h6 class="mb-0"><i class="fas fa-database me-2"></i>Información del Sistema</h6>
                            </div>
                            <div class="card-body">
                                <ul class="list-unstyled mb-0">
                                    <li class="mb-2">
                                        <i class="fas fa-server text-primary me-2"></i>
                                        <strong>PHP Version:</strong> <?php echo phpversion(); ?>
                                    </li>
                                    <li class="mb-2">
                                        <i class="fas fa-database text-primary me-2"></i>
                                        <strong>MySQL Version:</strong> 
                                        <?php 
                                        try {
                                            $version = $db->getAttribute(PDO::ATTR_SERVER_VERSION);
                                            echo $version;
                                        } catch (Exception $e) {
                                            echo 'No disponible';
                                        }
                                        ?>
                                    </li>
                                    <li class="mb-2">
                                        <i class="fas fa-folder text-primary me-2"></i>
                                        <strong>Directorio:</strong> <?php echo __DIR__; ?>
                                    </li>
                                </ul>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header">
                                <h6 class="mb-0"><i class="fas fa-shield-alt me-2"></i>Recomendaciones</h6>
                            </div>
                            <div class="card-body">
                                <div class="alert alert-warning mb-2">
                                    <small><i class="fas fa-exclamation-triangle me-1"></i> Mantenga siempre una copia de seguridad de su logo.</small>
                                </div>
                                <div class="alert alert-info mb-2">
                                    <small><i class="fas fa-info-circle me-1"></i> Los campos marcados con * son obligatorios.</small>
                                </div>
                                <div class="alert alert-success mb-0">
                                    <small><i class="fas fa-check-circle me-1"></i> Los cambios se aplican inmediatamente.</small>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Vista previa del logo
        document.getElementById('logo').addEventListener('change', function(e) {
            const file = e.target.files[0];
            const preview = document.getElementById('logoPreview');
            
            if (file) {
                const reader = new FileReader();
                
                reader.onload = function(e) {
                    // Limpiar contenido previo
                    preview.innerHTML = '';
                    
                    // Crear nueva imagen
                    const img = document.createElement('img');
                    img.src = e.target.result;
                    img.alt = 'Vista previa del logo';
                    img.style.maxWidth = '100%';
                    img.style.maxHeight = '100%';
                    img.style.objectFit = 'contain';
                    
                    preview.appendChild(img);
                };
                
                reader.readAsDataURL(file);
            }
        });
        
        // Validar tamaño de archivo
        document.getElementById('formConfig').addEventListener('submit', function(e) {
            const fileInput = document.getElementById('logo');
            const maxSize = 2 * 1024 * 1024; // 2MB
            
            if (fileInput.files.length > 0) {
                const fileSize = fileInput.files[0].size;
                
                if (fileSize > maxSize) {
                    e.preventDefault();
                    alert('El archivo es demasiado grande. Tamaño máximo: 2MB');
                    fileInput.value = '';
                    return false;
                }
                
                // Validar extensión
                const fileName = fileInput.files[0].name;
                const extension = fileName.split('.').pop().toLowerCase();
                const allowedExtensions = ['jpg', 'jpeg', 'png', 'gif', 'svg'];
                
                if (!allowedExtensions.includes(extension)) {
                    e.preventDefault();
                    alert('Formato de archivo no permitido. Use: ' + allowedExtensions.join(', '));
                    fileInput.value = '';
                    return false;
                }
            }
            
            // Validar campos obligatorios
            const codigo = document.getElementById('codigo').value.trim();
            const nombre = document.getElementById('nombre_completo').value.trim();
            
            if (!codigo || !nombre) {
                e.preventDefault();
                alert('Los campos Código y Nombre Completo son obligatorios');
                return false;
            }
            
            return true;
        });
        
        // Mostrar confirmación antes de salir si hay cambios sin guardar
        let formChanged = false;
        const form = document.getElementById('formConfig');
        const inputs = form.querySelectorAll('input, textarea, select');
        
        inputs.forEach(input => {
            input.addEventListener('change', () => {
                formChanged = true;
            });
        });
        
        window.addEventListener('beforeunload', (e) => {
            if (formChanged) {
                e.preventDefault();
                e.returnValue = '';
            }
        });
        
        // Resetear bandera de cambios al enviar el formulario
        form.addEventListener('submit', () => {
            formChanged = false;
        });
    </script>
</body>
</html>