<?php
// Habilitar errores temporalmente
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

session_start();

// Verificar sesión
if (!isset($_SESSION['user_id'])) {
    header("Location: ../../login.php");
    exit();
}

require_once '../../config/database.php';

$database = new Database();
$db = $database->getConnection();

if (!$db) {
    die("Error de conexión a la base de datos");
}

// Obtener áreas para filtros
$areas = [];
try {
    $query = "SELECT id, nombre_area FROM areas WHERE activo = 1 ORDER BY nombre_area";
    $stmt = $db->prepare($query);
    $stmt->execute();
    $areas = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    error_log("Error al cargar áreas: " . $e->getMessage());
}

// Variables para filtros
$fecha_inicio = $_GET['fecha_inicio'] ?? date('Y-m-d', strtotime('-7 days'));
$fecha_fin = $_GET['fecha_fin'] ?? date('Y-m-d');
$dni = $_GET['dni'] ?? '';
$area_id = $_GET['area_id'] ?? '';
$tipo_reporte = $_GET['tipo_reporte'] ?? 'diario';

// Obtener datos filtrados
$visitas = [];
$total_visitas = 0;
$total_activas = 0;

try {
    $query = "SELECT v.*, a.nombre_area as area_nombre, 
              CONCAT(u.nombres, ' ', u.apellidos) as usuario_registra
              FROM visitas v 
              LEFT JOIN areas a ON v.area_id = a.id 
              LEFT JOIN usuarios u ON v.usuario_registra_id = u.id 
              WHERE DATE(v.fecha_visita) BETWEEN :fecha_inicio AND :fecha_fin";
    
    $params = [
        ':fecha_inicio' => $fecha_inicio,
        ':fecha_fin' => $fecha_fin
    ];
    
    // Aplicar filtros adicionales
    if (!empty($dni)) {
        $query .= " AND v.documento_visitante LIKE :dni";
        $params[':dni'] = "%$dni%";
    }
    
    if (!empty($area_id) && $area_id != 'todas') {
        $query .= " AND v.area_id = :area_id";
        $params[':area_id'] = $area_id;
    }
    
    // Aplicar filtro por tipo de reporte
    switch ($tipo_reporte) {
        case 'diario':
            $query .= " AND DATE(v.fecha_visita) = CURDATE()";
            break;
        case 'mensual':
            $query .= " AND MONTH(v.fecha_visita) = MONTH(CURDATE()) 
                       AND YEAR(v.fecha_visita) = YEAR(CURDATE())";
            break;
        case 'activas':
            $query .= " AND v.hora_salida IS NULL";
            break;
        case 'completadas':
            $query .= " AND v.hora_salida IS NOT NULL";
            break;
    }
    
    $query .= " ORDER BY v.fecha_visita DESC, v.hora_ingreso DESC";
    
    $stmt = $db->prepare($query);
    
    foreach ($params as $key => $value) {
        $stmt->bindValue($key, $value);
    }
    
    $stmt->execute();
    $visitas = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Calcular estadísticas
    $total_visitas = count($visitas);
    foreach ($visitas as $visita) {
        if ($visita['hora_salida'] === null) {
            $total_activas++;
        }
    }
    
} catch (PDOException $e) {
    error_log("Error al cargar visitas: " . $e->getMessage());
}

// Generar PDF si se solicita
if (isset($_GET['exportar']) && $_GET['exportar'] == 'pdf') {
    require_once '../../lib/fpdf/fpdf.php';
    
    class PDF extends FPDF {
        private $entidad_data;
        private $usuario_data;
        private $filtros_data;
        
        function __construct($entidad, $usuario, $filtros) {
            parent::__construct('L'); // Landscape orientation
            $this->entidad_data = $entidad;
            $this->usuario_data = $usuario;
            $this->filtros_data = $filtros;
        }
        
        // Cabecera de página
        function Header() {
            // Logo de la entidad
            $logo_path = '../../assets/img/' . ($this->entidad_data['logo'] ?? '');
            if (!empty($this->entidad_data['logo']) && file_exists($logo_path)) {
                $this->Image($logo_path, 10, 8, 25);
                $this->SetX(40);
            } else {
                $this->SetX(10);
            }
            
            // Título principal
            $this->SetFont('Arial', 'B', 16);
            $this->Cell(0, 8, 'REPORTE DE VISITAS', 0, 1, 'C');
            
            // Nombre de la entidad
            $this->SetFont('Arial', 'B', 12);
            $this->Cell(0, 6, strtoupper($this->entidad_data['nombre_completo'] ?? 'ENTIDAD PÚBLICA'), 0, 1, 'C');
            
            // Línea separadora
            $this->SetLineWidth(0.5);
            $this->SetDrawColor(25, 118, 210); // Celeste oscuro
            $this->Line(10, 30, 287, 30);
            
            // Información del reporte
            $this->SetY(35);
            $this->SetFont('Arial', '', 10);
            
            // Filtros aplicados
            $this->Cell(0, 5, 'PERIODO: ' . date('d/m/Y', strtotime($this->filtros_data['fecha_inicio'])) . ' - ' . 
                           date('d/m/Y', strtotime($this->filtros_data['fecha_fin'])), 0, 1);
            
            if (!empty($this->filtros_data['dni'])) {
                $this->Cell(0, 5, 'DNI: ' . $this->filtros_data['dni'], 0, 1);
            }
            
            if ($this->filtros_data['area_id'] != 'todas' && !empty($this->filtros_data['area_nombre'])) {
                $this->Cell(0, 5, 'AREA: ' . $this->filtros_data['area_nombre'], 0, 1);
            }
            
            if ($this->filtros_data['tipo_reporte'] != 'todos') {
                $tipos = [
                    'diario' => 'VISITAS DE HOY',
                    'mensual' => 'VISITAS DEL MES',
                    'activas' => 'VISITAS ACTIVAS',
                    'completadas' => 'VISITAS COMPLETADAS'
                ];
                $this->Cell(0, 5, 'TIPO: ' . ($tipos[$this->filtros_data['tipo_reporte']] ?? ''), 0, 1);
            }
            
            $this->Cell(0, 5, 'FECHA DE GENERACION: ' . date('d/m/Y H:i:s'), 0, 1);
            $this->Ln(5);
        }
        
        // Pie de página
        function Footer() {
            // Posición: a 1.5 cm del final
            $this->SetY(-15);
            
            // Línea separadora
            $this->SetLineWidth(0.3);
            $this->SetDrawColor(150, 150, 150);
            $this->Line(10, $this->GetY(), 287, $this->GetY());
            
            // Información del usuario
            $this->SetFont('Arial', 'I', 8);
            $this->Cell(0, 5, 'Generado por: ' . $this->usuario_data['nombre'], 0, 0, 'L');
            
            // Número de página
            $this->Cell(0, 5, 'Pagina ' . $this->PageNo() . '/{nb}', 0, 0, 'R');
            
            // Información del sistema
            $this->SetY(-10);
            $this->SetFont('Arial', 'I', 7);
            $this->Cell(0, 5, 'Sistema de Control de Visitas - ' . date('Y'), 0, 0, 'C');
        }
        
        // Tabla mejorada con colores
        function ImprovedTable($header, $data, $total_visitas, $total_activas) {
            // Colores, ancho de línea y fuente en negrita
            $this->SetFillColor(25, 118, 210); // Celeste oscuro
            $this->SetTextColor(255);
            $this->SetDrawColor(150, 150, 150);
            $this->SetLineWidth(0.3);
            $this->SetFont('Arial', 'B', 9);
            
            // Anchuras de las columnas (para orientación landscape)
            $w = array(20, 30, 50, 20, 40, 50, 20, 30, 27);
            
            // Cabeceras
            for($i = 0; $i < count($header); $i++)
                $this->Cell($w[$i], 7, $header[$i], 1, 0, 'C', true);
            $this->Ln();
            
            // Restauración de colores y fuentes
            $this->SetFillColor(224, 235, 255);
            $this->SetTextColor(0);
            $this->SetFont('Arial', '', 8);
            
            // Datos
            $fill = false;
            foreach($data as $row) {
                // Verificar si necesita nueva página
                if($this->GetY() > 170) {
                    $this->AddPage();
                    // Volver a dibujar cabecera de tabla
                    $this->SetFillColor(25, 118, 210);
                    $this->SetTextColor(255);
                    $this->SetFont('Arial', 'B', 9);
                    for($i = 0; $i < count($header); $i++)
                        $this->Cell($w[$i], 7, $header[$i], 1, 0, 'C', true);
                    $this->Ln();
                    $this->SetFillColor(224, 235, 255);
                    $this->SetTextColor(0);
                    $this->SetFont('Arial', '', 8);
                    $fill = false;
                }
                
                for($i = 0; $i < count($row); $i++) {
                    // Alternar color de fondo
                    $this->Cell($w[$i], 6, $row[$i], 'LR', 0, 'C', $fill);
                }
                $this->Ln();
                $fill = !$fill;
            }
            
            // Línea de cierre
            $this->Cell(array_sum($w), 0, '', 'T');
            
            // Estadísticas al final
            $this->Ln(5);
            $this->SetFont('Arial', 'B', 10);
            $this->Cell(0, 8, 'RESUMEN ESTADISTICO', 0, 1, 'C');
            
            $this->SetFont('Arial', '', 9);
            $this->Cell(0, 6, 'Total de visitas: ' . $total_visitas, 0, 1, 'C');
            $this->Cell(0, 6, 'Visitas activas: ' . $total_activas, 0, 1, 'C');
            $this->Cell(0, 6, 'Visitas completadas: ' . ($total_visitas - $total_activas), 0, 1, 'C');
            
            if ($total_visitas > 0) {
                $porcentaje = round(($total_activas / $total_visitas) * 100, 1);
                $this->Cell(0, 6, 'Porcentaje de visitas activas: ' . $porcentaje . '%', 0, 1, 'C');
            }
        }
    }
    
    // Obtener datos de la entidad
    $entidad = [];
    try {
        $query = "SELECT * FROM entidad LIMIT 1";
        $stmt = $db->prepare($query);
        $stmt->execute();
        $entidad = $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error al cargar datos de entidad: " . $e->getMessage());
    }
    
    // Obtener nombre del área si se filtró por área
    $area_nombre = '';
    if (!empty($area_id) && $area_id != 'todas') {
        try {
            $query = "SELECT nombre_area FROM areas WHERE id = :area_id";
            $stmt = $db->prepare($query);
            $stmt->bindParam(':area_id', $area_id);
            $stmt->execute();
            $area = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($area) {
                $area_nombre = $area['nombre_area'];
            }
        } catch (PDOException $e) {
            error_log("Error al obtener nombre del área: " . $e->getMessage());
        }
    }
    
    // Datos del usuario que genera el reporte
    $usuario_genera = [
        'nombre' => $_SESSION['user_nombre'] ?? 'Usuario no identificado',
        'dni' => $_SESSION['user_dni'] ?? 'N/A',
        'tipo' => $_SESSION['user_tipo'] ?? 'N/A'
    ];
    
    // Datos de filtros para el encabezado
    $filtros_pdf = [
        'fecha_inicio' => $fecha_inicio,
        'fecha_fin' => $fecha_fin,
        'dni' => $dni,
        'area_id' => $area_id,
        'area_nombre' => $area_nombre,
        'tipo_reporte' => $tipo_reporte
    ];
    
    // Crear PDF
    $pdf = new PDF($entidad, $usuario_genera, $filtros_pdf);
    $pdf->AliasNbPages();
    $pdf->AddPage();
    
    // Cabecera de la tabla
    $header = array('FECHA', 'HORA ING.', 'HORA SAL.', 'VISITANTE', 'DNI', 'AREA', 'MOTIVO', 'ESTADO', 'REGISTRADO POR');
    
    // Preparar datos para la tabla
    $data = array();
    foreach ($visitas as $visita) {
        // Formatear estado
        $estado = $visita['hora_salida'] ? 'COMPLETADA' : 'ACTIVA';
        
        // Formatear horas
        $hora_ingreso = date('H:i', strtotime($visita['hora_ingreso']));
        $hora_salida = $visita['hora_salida'] ? date('H:i', strtotime($visita['hora_salida'])) : 'PENDIENTE';
        
        // Acortar motivo si es muy largo
        $motivo = $visita['motivo_visita'];
        if (strlen($motivo) > 40) {
            $motivo = substr($motivo, 0, 37) . '...';
        }
        
        $data[] = array(
            date('d/m/Y', strtotime($visita['fecha_visita'])),
            $hora_ingreso,
            $hora_salida,
            $visita['apellidos_visitante'] . ', ' . $visita['nombres_visitante'],
            $visita['documento_visitante'],
            $visita['area_nombre'] ?? 'NO ESPECIFICADA',
            $motivo,
            $estado,
            $visita['usuario_registra'] ?? 'NO REGISTRADO'
        );
    }
    
    // Crear tabla
    $pdf->SetFont('Arial', '', 9);
    $pdf->ImprovedTable($header, $data, $total_visitas, $total_activas);
    
    // Generar nombre de archivo
    $fecha_reporte = date('Ymd_His');
    $filename = 'reporte_visitas_' . $fecha_reporte . '.pdf';
    
    // Salida del PDF (I = ver en navegador, D = descargar)
    $pdf->Output('I', $filename);
    exit();
}
//fin genrar PDF

 
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reportes - Sistema de Visitas</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <link rel="stylesheet" href="https://cdn.datatables.net/1.11.5/css/dataTables.bootstrap5.min.css">
    <style>
        :root {
            --celeste-primario: #e3f2fd;
            --celeste-secundario: #bbdefb;
            --celeste-oscuro: #1976d2;
        }
        body {
            background-color: #f8f9fa;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        .main-container {
            margin-left: 250px;
            padding: 20px;
            min-height: 100vh;
        }
        .sidebar {
            background: linear-gradient(180deg, var(--celeste-oscuro) 0%, #1565c0 100%);
            min-height: 100vh;
            color: white;
            padding: 0;
            position: fixed;
            width: 250px;
        }
        .sidebar .nav-link {
            color: rgba(255,255,255,0.8);
            padding: 12px 20px;
            margin: 2px 10px;
            border-radius: 8px;
            transition: all 0.3s;
            text-decoration: none;
        }
        .sidebar .nav-link:hover, .sidebar .nav-link.active {
            background: rgba(255,255,255,0.1);
            color: white;
            transform: translateX(5px);
        }
        .card {
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.05);
            border: none;
            margin-bottom: 20px;
        }
        .card-header {
            background: linear-gradient(to right, #6f42c1, #9c27b0);
            color: white;
            border-radius: 15px 15px 0 0 !important;
            padding: 15px 20px;
            font-weight: 600;
        }
        .btn-primary {
            background: linear-gradient(to right, var(--celeste-oscuro), #2196f3);
            border: none;
            border-radius: 8px;
            padding: 10px 20px;
            transition: all 0.3s;
        }
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(33, 150, 243, 0.4);
        }
        .btn-success {
            background: linear-gradient(to right, #28a745, #20c997);
            border: none;
            border-radius: 8px;
        }
        .btn-warning {
            background: linear-gradient(to right, #ff9800, #ffb74d);
            border: none;
            border-radius: 8px;
        }
        .stat-card {
            background: white;
            border-radius: 10px;
            padding: 20px;
            text-align: center;
            box-shadow: 0 3px 10px rgba(0,0,0,0.05);
            transition: transform 0.3s;
        }
        .stat-card:hover {
            transform: translateY(-5px);
        }
        .stat-card .stat-number {
            font-size: 2.5rem;
            font-weight: bold;
            margin: 10px 0;
        }
        .stat-card.total {
            border-top: 5px solid var(--celeste-oscuro);
        }
        .stat-card.activas {
            border-top: 5px solid #ff9800;
        }
        .filter-card {
            background: white;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
        }
        .badge-estado {
            padding: 5px 10px;
            border-radius: 20px;
            font-size: 0.85em;
        }
        .badge-activa {
            background: #fff3cd;
            color: #856404;
        }
        .badge-completada {
            background: #d4edda;
            color: #155724;
        }
        @media (max-width: 768px) {
            .sidebar {
                width: 100%;
                position: relative;
                min-height: auto;
            }
            .main-container {
                margin-left: 0;
            }
        }
        #tablaReportes {
            border-radius: 10px;
            overflow: hidden;
        }
        #tablaReportes thead {
            background: linear-gradient(to right, var(--celeste-oscuro), #2196f3);
            color: white;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 col-lg-2 d-md-block sidebar">
                <div class="position-sticky pt-3">
                    <div class="text-center mb-4 p-3">
                        <div class="p-3">
                            <i class="fas fa-building fa-3x mb-3"></i>
                            <h5>Sistema de Visitas</h5>
                            <small>Entidad Pública</small>
                        </div>
                    </div>
                    
                    <div class="nav flex-column">
                        <a class="nav-link" href="../../dashboard.php">
                            <i class="fas fa-home me-2"></i>Dashboard
                        </a>
                        <a class="nav-link" href="../visitas/registrar.php">
                            <i class="fas fa-sign-in-alt me-2"></i>Registrar Ingreso
                        </a>
                        <a class="nav-link" href="../visitas/salida.php">
                            <i class="fas fa-sign-out-alt me-2"></i>Registrar Salida
                        </a>
                        <a class="nav-link" href="../visitas/listar.php">
                            <i class="fas fa-list me-2"></i>Listar Visitas
                        </a>
                        <a class="nav-link active" href="generar.php">
                            <i class="fas fa-chart-bar me-2"></i>Reportes
                        </a>
                        
                        <?php if (isset($_SESSION['user_tipo']) && $_SESSION['user_tipo'] == 'admin'): ?>
                        <hr class="my-3 text-white-50">
                        <h6 class="sidebar-heading px-3 mt-4 mb-1 text-white-50">
                            <span>Administración</span>
                        </h6>
                        <a class="nav-link" href="../usuarios/gestion.php">
                            <i class="fas fa-users me-2"></i>Gestión de Usuarios
                        </a>
                        <a class="nav-link" href="../areas/gestion.php">
                            <i class="fas fa-sitemap me-2"></i>Áreas
                        </a>
                        <a class="nav-link" href="../entidad/configuracion.php">
                            <i class="fas fa-cog me-2"></i>Configuración
                        </a>
                        <?php endif; ?>
                        
                        <hr class="my-3 text-white-50">
                        <a class="nav-link" href="../../logout.php">
                            <i class="fas fa-sign-out-alt me-2"></i>Cerrar Sesión
                        </a>
                    </div>
                </div>
            </div>

            <!-- Main Content -->
            <div class="col-md-9 ms-sm-auto col-lg-10 px-md-4 main-container">
                <!-- Header -->
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Reportes de Visitas</h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <span class="badge bg-primary">
                            <i class="fas fa-user me-1"></i>
                            <?php echo isset($_SESSION['user_nombre']) ? htmlspecialchars($_SESSION['user_nombre']) : 'Usuario'; ?>
                        </span>
                    </div>
                </div>
                
                <!-- Filtros -->
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="fas fa-filter me-2"></i>Filtros de Búsqueda</h5>
                    </div>
                    <div class="card-body">
                        <form method="GET" action="" id="formFiltros">
                            <div class="row">
                                <div class="col-md-3 mb-3">
                                    <label for="fecha_inicio" class="form-label">Fecha Inicio</label>
                                    <input type="date" class="form-control" id="fecha_inicio" name="fecha_inicio" 
                                           value="<?php echo htmlspecialchars($fecha_inicio); ?>" required>
                                </div>
                                
                                <div class="col-md-3 mb-3">
                                    <label for="fecha_fin" class="form-label">Fecha Fin</label>
                                    <input type="date" class="form-control" id="fecha_fin" name="fecha_fin" 
                                           value="<?php echo htmlspecialchars($fecha_fin); ?>" required>
                                </div>
                                
                                <div class="col-md-3 mb-3">
                                    <label for="dni" class="form-label">DNI Visitante</label>
                                    <input type="text" class="form-control" id="dni" name="dni" 
                                           value="<?php echo htmlspecialchars($dni); ?>" 
                                           placeholder="Buscar por DNI">
                                </div>
                                
                                <div class="col-md-3 mb-3">
                                    <label for="area_id" class="form-label">Área</label>
                                    <select class="form-select" id="area_id" name="area_id">
                                        <option value="todas">Todas las áreas</option>
                                        <?php foreach ($areas as $area): ?>
                                            <option value="<?php echo $area['id']; ?>" 
                                                <?php echo ($area_id == $area['id']) ? 'selected' : ''; ?>>
                                                <?php echo htmlspecialchars($area['nombre_area']); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="tipo_reporte" class="form-label">Tipo de Reporte</label>
                                    <select class="form-select" id="tipo_reporte" name="tipo_reporte">
                                        <option value="todos" <?php echo ($tipo_reporte == 'todos') ? 'selected' : ''; ?>>Todos los registros</option>
                                        <option value="diario" <?php echo ($tipo_reporte == 'diario') ? 'selected' : ''; ?>>Visitas de hoy</option>
                                        <option value="mensual" <?php echo ($tipo_reporte == 'mensual') ? 'selected' : ''; ?>>Visitas del mes</option>
                                        <option value="activas" <?php echo ($tipo_reporte == 'activas') ? 'selected' : ''; ?>>Visitas activas</option>
                                        <option value="completadas" <?php echo ($tipo_reporte == 'completadas') ? 'selected' : ''; ?>>Visitas completadas</option>
                                    </select>
                                </div>
                                
                                <div class="col-md-6 mb-3 d-flex align-items-end">
                                    <div class="btn-group w-100">
                                        <button type="submit" class="btn btn-primary flex-fill">
                                            <i class="fas fa-search me-2"></i>Buscar
                                        </button>
                                        <a href="generar.php" class="btn btn-secondary flex-fill">
                                            <i class="fas fa-redo me-2"></i>Limpiar
                                        </a>
                                    </div>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>
                
                <!-- Estadísticas -->
                <div class="row mb-4">
                    <div class="col-md-4">
                        <div class="stat-card total">
                            <i class="fas fa-users fa-2x text-primary mb-2"></i>
                            <h5>Total Visitas</h5>
                            <div class="stat-number text-primary"><?php echo $total_visitas; ?></div>
                            <small>En el periodo seleccionado</small>
                        </div>
                    </div>
                    
                    <div class="col-md-4">
                        <div class="stat-card activas">
                            <i class="fas fa-user-clock fa-2x text-warning mb-2"></i>
                            <h5>Visitas Activas</h5>
                            <div class="stat-number text-warning"><?php echo $total_activas; ?></div>
                            <small>Sin registrar salida</small>
                        </div>
                    </div>
                    
                    <div class="col-md-4">
                        <div class="stat-card">
                            <i class="fas fa-chart-line fa-2x text-success mb-2"></i>
                            <h5>Porcentaje</h5>
                            <div class="stat-number text-success">
                                <?php echo $total_visitas > 0 ? round(($total_activas / $total_visitas) * 100, 1) : 0; ?>%
                            </div>
                            <small>Visitas activas</small>
                        </div>
                    </div>
                </div>
                
                <!-- Tabla de resultados -->
                <div class="card">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h5 class="mb-0">
                            <i class="fas fa-table me-2"></i>Resultados del Reporte
                        </h5>
                        <div>
                            <?php if ($total_visitas > 0): ?>
                            <a href="generar.php?<?php echo http_build_query($_GET); ?>&exportar=pdf" 
                               class="btn btn-danger btn-sm" target="_blank">
                                <i class="fas fa-file-pdf me-1"></i>Exportar PDF
                            </a>
                            <button class="btn btn-success btn-sm" onclick="exportarExcel()">
                                <i class="fas fa-file-excel me-1"></i>Exportar Excel
                            </button>
                            <?php endif; ?>
                        </div>
                    </div>
                    <div class="card-body">
                        <?php if ($total_visitas > 0): ?>
                        <div class="table-responsive">
                            <table id="tablaReportes" class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Fecha</th>
                                        <th>Hora Ingreso</th>
                                        <th>Hora Salida</th>
                                        <th>Visitante</th>
                                        <th>DNI</th>
                                        <th>Área</th>
                                        <th>Motivo</th>
                                        <th>Estado</th>
                                        <th>Registrado por</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($visitas as $visita): ?>
                                    <tr>
                                        <td><?php echo date('d/m/Y', strtotime($visita['fecha_visita'])); ?></td>
                                        <td class="fw-bold"><?php echo date('H:i', strtotime($visita['hora_ingreso'])); ?></td>
                                        <td>
                                            <?php if ($visita['hora_salida']): ?>
                                                <span class="text-success fw-bold">
                                                    <?php echo date('H:i', strtotime($visita['hora_salida'])); ?>
                                                </span>
                                            <?php else: ?>
                                                <span class="text-muted">Pendiente</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php echo htmlspecialchars($visita['apellidos_visitante'] . ', ' . $visita['nombres_visitante']); ?>
                                        </td>
                                        <td>
                                            <span class="badge bg-secondary">
                                                <?php echo htmlspecialchars($visita['documento_visitante']); ?>
                                            </span>
                                        </td>
                                        <td><?php echo htmlspecialchars($visita['area_nombre'] ?? 'No especificada'); ?></td>
                                        <td>
                                            <span data-bs-toggle="tooltip" title="<?php echo htmlspecialchars($visita['motivo_visita']); ?>">
                                                <?php echo htmlspecialchars(substr($visita['motivo_visita'], 0, 40)) . '...'; ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php if ($visita['hora_salida']): ?>
                                                <span class="badge badge-estado badge-completada">
                                                    <i class="fas fa-check-circle me-1"></i>Completada
                                                </span>
                                            <?php else: ?>
                                                <span class="badge badge-estado badge-activa">
                                                    <i class="fas fa-clock me-1"></i>Activa
                                                </span>
                                            <?php endif; ?>
                                        </td>
                                        <td><?php echo htmlspecialchars($visita['usuario_registra'] ?? 'No registrado'); ?></td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        
                        <div class="mt-3 text-muted">
                            <small>
                                <i class="fas fa-info-circle me-1"></i>
                                Mostrando <?php echo $total_visitas; ?> registros del 
                                <?php echo date('d/m/Y', strtotime($fecha_inicio)); ?> al 
                                <?php echo date('d/m/Y', strtotime($fecha_fin)); ?>
                            </small>
                        </div>
                        <?php else: ?>
                        <div class="text-center py-5">
                            <i class="fas fa-search fa-4x text-muted mb-3"></i>
                            <h4 class="mb-3">No se encontraron resultados</h4>
                            <p class="text-muted mb-4">No hay visitas que coincidan con los criterios de búsqueda.</p>
                            <a href="generar.php" class="btn btn-primary">
                                <i class="fas fa-redo me-2"></i>Ver todas las visitas
                            </a>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.datatables.net/1.11.5/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.11.5/js/dataTables.bootstrap5.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/xlsx/0.18.5/xlsx.full.min.js"></script>
    
    <script>
        // Inicializar DataTable
        $(document).ready(function() {
            $('#tablaReportes').DataTable({
                language: {
                    url: '//cdn.datatables.net/plug-ins/1.11.5/i18n/es-ES.json'
                },
                pageLength: 25,
                order: [[0, 'desc']],
                dom: '<"row"<"col-md-6"l><"col-md-6"f>>rt<"row"<"col-md-6"i><"col-md-6"p>>'
            });
            
            // Inicializar tooltips
            var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
            var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
                return new bootstrap.Tooltip(tooltipTriggerEl);
            });
        });
        
        // Función para exportar a Excel
        function exportarExcel() {
            // Crear una nueva hoja de trabajo
            var wb = XLSX.utils.book_new();
            
            // Obtener datos de la tabla
            var table = document.getElementById('tablaReportes');
            var ws = XLSX.utils.table_to_sheet(table);
            
            // Añadir hoja al libro
            XLSX.utils.book_append_sheet(wb, ws, "Reporte Visitas");
            
            // Generar nombre de archivo
            var fecha = new Date().toISOString().slice(0,10);
            var filename = "reporte_visitas_" + fecha + ".xlsx";
            
            // Descargar archivo
            XLSX.writeFile(wb, filename);
        }
        
        // Validar que fecha fin no sea menor a fecha inicio
        document.getElementById('formFiltros').addEventListener('submit', function(e) {
            var fechaInicio = new Date(document.getElementById('fecha_inicio').value);
            var fechaFin = new Date(document.getElementById('fecha_fin').value);
            
            if (fechaFin < fechaInicio) {
                e.preventDefault();
                alert('La fecha fin no puede ser menor a la fecha inicio');
                document.getElementById('fecha_fin').focus();
                return false;
            }
            
            // Validar que no sean fechas futuras
            var hoy = new Date();
            hoy.setHours(0,0,0,0);
            
            if (fechaInicio > hoy) {
                e.preventDefault();
                alert('La fecha inicio no puede ser futura');
                document.getElementById('fecha_inicio').focus();
                return false;
            }
            
            if (fechaFin > hoy) {
                e.preventDefault();
                alert('La fecha fin no puede ser futura');
                document.getElementById('fecha_fin').focus();
                return false;
            }
            
            return true;
        });
        
        // Setear fecha máxima como hoy
        document.addEventListener('DOMContentLoaded', function() {
            var hoy = new Date().toISOString().split('T')[0];
            document.getElementById('fecha_inicio').max = hoy;
            document.getElementById('fecha_fin').max = hoy;
            
            // Si fecha fin está vacía, setear a hoy
            if (!document.getElementById('fecha_fin').value) {
                document.getElementById('fecha_fin').value = hoy;
            }
        });
    </script>
</body>
</html>