<?php
// Habilitar errores temporalmente
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

session_start();

// Verificar sesión y permisos de administrador
if (!isset($_SESSION['user_id'])) {
    header("Location: ../../login.php");
    exit();
}

// Verificar que sea administrador
if (!isset($_SESSION['user_tipo']) || $_SESSION['user_tipo'] != 'admin') {
    $_SESSION['mensaje'] = "No tienes permisos para acceder a esta sección";
    $_SESSION['tipo_mensaje'] = "danger";
    header("Location: ../../dashboard.php");
    exit();
}

require_once '../../config/database.php';

$database = new Database();
$db = $database->getConnection();

if (!$db) {
    die("Error de conexión a la base de datos");
}

// Obtener áreas para el formulario
$areas = [];
try {
    $query = "SELECT id, nombre_area FROM areas WHERE activo = 1 ORDER BY nombre_area";
    $stmt = $db->prepare($query);
    $stmt->execute();
    $areas = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    error_log("Error al cargar áreas: " . $e->getMessage());
}

// Variables
$mensaje = '';
$tipo_mensaje = '';
$usuario_editar = null;

// Procesar acciones
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    if (isset($_POST['accion'])) {
        switch ($_POST['accion']) {
            case 'crear':
                $dni = trim($_POST['dni'] ?? '');
                $apellidos = trim($_POST['apellidos'] ?? '');
                $nombres = trim($_POST['nombres'] ?? '');
                $password = trim($_POST['password'] ?? '');
                $confirm_password = trim($_POST['confirm_password'] ?? '');
                $correo = trim($_POST['correo'] ?? '');
                $area_id = $_POST['area_id'] ?? 0;
                $tipo_usuario = $_POST['tipo_usuario'] ?? 'registrador';
                
                // Validaciones
                $errores = [];
                
                if (empty($dni)) {
                    $errores[] = "El DNI es obligatorio";
                }
                
                if (empty($apellidos) || empty($nombres)) {
                    $errores[] = "Los nombres y apellidos son obligatorios";
                }
                
                if (empty($password)) {
                    $errores[] = "La contraseña es obligatoria";
                } elseif (strlen($password) < 6) {
                    $errores[] = "La contraseña debe tener al menos 6 caracteres";
                } elseif ($password !== $confirm_password) {
                    $errores[] = "Las contraseñas no coinciden";
                }
                
                if (empty($correo) || !filter_var($correo, FILTER_VALIDATE_EMAIL)) {
                    $errores[] = "El correo electrónico es inválido";
                }
                
                if (empty($area_id)) {
                    $errores[] = "Debe seleccionar un área";
                }
                
                if (empty($errores)) {
                    try {
                        // Verificar si el DNI ya existe
                        $query = "SELECT id FROM usuarios WHERE dni = :dni";
                        $stmt = $db->prepare($query);
                        $stmt->bindParam(':dni', $dni);
                        $stmt->execute();
                        
                        if ($stmt->rowCount() > 0) {
                            $mensaje = "Ya existe un usuario con ese DNI";
                            $tipo_mensaje = "danger";
                        } else {
                            // Verificar si el correo ya existe
                            $query = "SELECT id FROM usuarios WHERE correo = :correo";
                            $stmt = $db->prepare($query);
                            $stmt->bindParam(':correo', $correo);
                            $stmt->execute();
                            
                            if ($stmt->rowCount() > 0) {
                                $mensaje = "Ya existe un usuario con ese correo electrónico";
                                $tipo_mensaje = "danger";
                            } else {
                                // Hash de la contraseña
                                $hashed_password = password_hash($password, PASSWORD_BCRYPT);
                                
                                // Generar token de verificación
                                $token_verificacion = bin2hex(random_bytes(32));
                                
                                $query = "INSERT INTO usuarios (dni, apellidos, nombres, password, correo, 
                                          area_id, tipo_usuario, token_verificacion, email_verificado) 
                                          VALUES (:dni, :apellidos, :nombres, :password, :correo, 
                                          :area_id, :tipo_usuario, :token, 1)";
                                
                                $stmt = $db->prepare($query);
                                $stmt->bindParam(':dni', $dni);
                                $stmt->bindParam(':apellidos', $apellidos);
                                $stmt->bindParam(':nombres', $nombres);
                                $stmt->bindParam(':password', $hashed_password);
                                $stmt->bindParam(':correo', $correo);
                                $stmt->bindParam(':area_id', $area_id, PDO::PARAM_INT);
                                $stmt->bindParam(':tipo_usuario', $tipo_usuario);
                                $stmt->bindParam(':token', $token_verificacion);
                                
                                if ($stmt->execute()) {
                                    $mensaje = "Usuario creado exitosamente";
                                    $tipo_mensaje = "success";
                                    
                                    // Limpiar formulario
                                    $_POST = array();
                                } else {
                                    $mensaje = "Error al crear el usuario";
                                    $tipo_mensaje = "danger";
                                }
                            }
                        }
                    } catch (PDOException $e) {
                        $mensaje = "Error: " . $e->getMessage();
                        $tipo_mensaje = "danger";
                        error_log("Error al crear usuario: " . $e->getMessage());
                    }
                } else {
                    $mensaje = implode("<br>", $errores);
                    $tipo_mensaje = "danger";
                }
                break;
                
            case 'editar':
                $id = $_POST['id'] ?? 0;
                $dni = trim($_POST['dni'] ?? '');
                $apellidos = trim($_POST['apellidos'] ?? '');
                $nombres = trim($_POST['nombres'] ?? '');
                $correo = trim($_POST['correo'] ?? '');
                $area_id = $_POST['area_id'] ?? 0;
                $tipo_usuario = $_POST['tipo_usuario'] ?? 'registrador';
                
                // Validaciones
                $errores = [];
                
                if (empty($dni)) {
                    $errores[] = "El DNI es obligatorio";
                }
                
                if (empty($apellidos) || empty($nombres)) {
                    $errores[] = "Los nombres y apellidos son obligatorios";
                }
                
                if (empty($correo) || !filter_var($correo, FILTER_VALIDATE_EMAIL)) {
                    $errores[] = "El correo electrónico es inválido";
                }
                
                if (empty($area_id)) {
                    $errores[] = "Debe seleccionar un área";
                }
                
                if (empty($errores)) {
                    try {
                        // Verificar si el DNI ya existe (excluyendo el usuario actual)
                        $query = "SELECT id FROM usuarios WHERE dni = :dni AND id != :id";
                        $stmt = $db->prepare($query);
                        $stmt->bindParam(':dni', $dni);
                        $stmt->bindParam(':id', $id, PDO::PARAM_INT);
                        $stmt->execute();
                        
                        if ($stmt->rowCount() > 0) {
                            $mensaje = "Ya existe otro usuario con ese DNI";
                            $tipo_mensaje = "danger";
                        } else {
                            // Verificar si el correo ya existe (excluyendo el usuario actual)
                            $query = "SELECT id FROM usuarios WHERE correo = :correo AND id != :id";
                            $stmt = $db->prepare($query);
                            $stmt->bindParam(':correo', $correo);
                            $stmt->bindParam(':id', $id, PDO::PARAM_INT);
                            $stmt->execute();
                            
                            if ($stmt->rowCount() > 0) {
                                $mensaje = "Ya existe otro usuario con ese correo electrónico";
                                $tipo_mensaje = "danger";
                            } else {
                                // Actualizar usuario
                                $query = "UPDATE usuarios SET 
                                          dni = :dni,
                                          apellidos = :apellidos,
                                          nombres = :nombres,
                                          correo = :correo,
                                          area_id = :area_id,
                                          tipo_usuario = :tipo_usuario
                                          WHERE id = :id";
                                
                                $stmt = $db->prepare($query);
                                $stmt->bindParam(':dni', $dni);
                                $stmt->bindParam(':apellidos', $apellidos);
                                $stmt->bindParam(':nombres', $nombres);
                                $stmt->bindParam(':correo', $correo);
                                $stmt->bindParam(':area_id', $area_id, PDO::PARAM_INT);
                                $stmt->bindParam(':tipo_usuario', $tipo_usuario);
                                $stmt->bindParam(':id', $id, PDO::PARAM_INT);
                                
                                if ($stmt->execute()) {
                                    $mensaje = "Usuario actualizado exitosamente";
                                    $tipo_mensaje = "success";
                                    $usuario_editar = null; // Limpiar edición
                                } else {
                                    $mensaje = "Error al actualizar el usuario";
                                    $tipo_mensaje = "danger";
                                }
                            }
                        }
                    } catch (PDOException $e) {
                        $mensaje = "Error: " . $e->getMessage();
                        $tipo_mensaje = "danger";
                        error_log("Error al editar usuario: " . $e->getMessage());
                    }
                } else {
                    $mensaje = implode("<br>", $errores);
                    $tipo_mensaje = "danger";
                }
                break;
                
            case 'cambiar_password':
                $id = $_POST['id'] ?? 0;
                $password = trim($_POST['password'] ?? '');
                $confirm_password = trim($_POST['confirm_password'] ?? '');
                
                if (empty($password)) {
                    $mensaje = "La contraseña es obligatoria";
                    $tipo_mensaje = "danger";
                } elseif (strlen($password) < 6) {
                    $mensaje = "La contraseña debe tener al menos 6 caracteres";
                    $tipo_mensaje = "danger";
                } elseif ($password !== $confirm_password) {
                    $mensaje = "Las contraseñas no coinciden";
                    $tipo_mensaje = "danger";
                } else {
                    try {
                        $hashed_password = password_hash($password, PASSWORD_BCRYPT);
                        
                        $query = "UPDATE usuarios SET password = :password WHERE id = :id";
                        $stmt = $db->prepare($query);
                        $stmt->bindParam(':password', $hashed_password);
                        $stmt->bindParam(':id', $id, PDO::PARAM_INT);
                        
                        if ($stmt->execute()) {
                            $mensaje = "Contraseña cambiada exitosamente";
                            $tipo_mensaje = "success";
                        } else {
                            $mensaje = "Error al cambiar la contraseña";
                            $tipo_mensaje = "danger";
                        }
                    } catch (PDOException $e) {
                        $mensaje = "Error: " . $e->getMessage();
                        $tipo_mensaje = "danger";
                    }
                }
                break;
                
            case 'eliminar':
                $id = $_POST['id'] ?? 0;
                
                try {
                    // No permitir eliminar al propio usuario
                    if ($id == $_SESSION['user_id']) {
                        $mensaje = "No puedes eliminar tu propio usuario";
                        $tipo_mensaje = "warning";
                    } else {
                        $query = "DELETE FROM usuarios WHERE id = :id";
                        $stmt = $db->prepare($query);
                        $stmt->bindParam(':id', $id, PDO::PARAM_INT);
                        
                        if ($stmt->execute()) {
                            $mensaje = "Usuario eliminado exitosamente";
                            $tipo_mensaje = "success";
                        } else {
                            $mensaje = "Error al eliminar el usuario";
                            $tipo_mensaje = "danger";
                        }
                    }
                } catch (PDOException $e) {
                    $mensaje = "Error: " . $e->getMessage();
                    $tipo_mensaje = "danger";
                }
                break;
                
            case 'toggle_estado':
                $id = $_POST['id'] ?? 0;
                
                try {
                    // No permitir desactivar al propio usuario
                    if ($id == $_SESSION['user_id']) {
                        $mensaje = "No puedes desactivar tu propio usuario";
                        $tipo_mensaje = "warning";
                    } else {
                        $query = "UPDATE usuarios SET activo = NOT activo WHERE id = :id";
                        $stmt = $db->prepare($query);
                        $stmt->bindParam(':id', $id, PDO::PARAM_INT);
                        
                        if ($stmt->execute()) {
                            $mensaje = "Estado del usuario actualizado";
                            $tipo_mensaje = "success";
                        }
                    }
                } catch (PDOException $e) {
                    $mensaje = "Error: " . $e->getMessage();
                    $tipo_mensaje = "danger";
                }
                break;
                
            case 'cargar_editar':
                $id = $_POST['id'] ?? 0;
                
                try {
                    $query = "SELECT * FROM usuarios WHERE id = :id";
                    $stmt = $db->prepare($query);
                    $stmt->bindParam(':id', $id, PDO::PARAM_INT);
                    $stmt->execute();
                    
                    if ($stmt->rowCount() > 0) {
                        $usuario_editar = $stmt->fetch(PDO::FETCH_ASSOC);
                    }
                } catch (PDOException $e) {
                    error_log("Error al cargar usuario: " . $e->getMessage());
                }
                break;
        }
    }
}

// Obtener usuarios
$usuarios = [];
try {
    $query = "SELECT u.*, a.nombre_area as area_nombre 
              FROM usuarios u 
              LEFT JOIN areas a ON u.area_id = a.id 
              ORDER BY u.apellidos, u.nombres";
    
    $stmt = $db->prepare($query);
    $stmt->execute();
    $usuarios = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    error_log("Error al cargar usuarios: " . $e->getMessage());
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestión de Usuarios - Sistema de Visitas</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <link rel="stylesheet" href="https://cdn.datatables.net/1.11.5/css/dataTables.bootstrap5.min.css">
    <style>
        :root {
            --celeste-primario: #e3f2fd;
            --celeste-secundario: #bbdefb;
            --celeste-oscuro: #1976d2;
        }
        body {
            background-color: #f8f9fa;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        .main-container {
            margin-left: 250px;
            padding: 20px;
            min-height: 100vh;
        }
        .sidebar {
            background: linear-gradient(180deg, var(--celeste-oscuro) 0%, #1565c0 100%);
            min-height: 100vh;
            color: white;
            padding: 0;
            position: fixed;
            width: 250px;
        }
        .sidebar .nav-link {
            color: rgba(255,255,255,0.8);
            padding: 12px 20px;
            margin: 2px 10px;
            border-radius: 8px;
            transition: all 0.3s;
            text-decoration: none;
        }
        .sidebar .nav-link:hover, .sidebar .nav-link.active {
            background: rgba(255,255,255,0.1);
            color: white;
            transform: translateX(5px);
        }
        .card {
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.05);
            border: none;
            margin-bottom: 20px;
        }
        .card-header {
            background: linear-gradient(to right, #6f42c1, #9c27b0);
            color: white;
            border-radius: 15px 15px 0 0 !important;
            padding: 15px 20px;
            font-weight: 600;
        }
        .btn-primary {
            background: linear-gradient(to right, var(--celeste-oscuro), #2196f3);
            border: none;
            border-radius: 8px;
            padding: 10px 20px;
            transition: all 0.3s;
        }
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(33, 150, 243, 0.4);
        }
        .btn-success {
            background: linear-gradient(to right, #28a745, #20c997);
            border: none;
            border-radius: 8px;
        }
        .btn-warning {
            background: linear-gradient(to right, #ff9800, #ffb74d);
            border: none;
            border-radius: 8px;
        }
        .badge-estado {
            padding: 5px 10px;
            border-radius: 20px;
            font-size: 0.85em;
        }
        .badge-activo {
            background: #d4edda;
            color: #155724;
        }
        .badge-inactivo {
            background: #f8d7da;
            color: #721c24;
        }
        .badge-admin {
            background: #d1ecf1;
            color: #0c5460;
        }
        .badge-registrador {
            background: #e2e3e5;
            color: #383d41;
        }
        .tab-content {
            border: 1px solid #dee2e6;
            border-top: none;
            border-radius: 0 0 15px 15px;
            padding: 20px;
            background: white;
        }
        .nav-tabs .nav-link {
            border-radius: 15px 15px 0 0;
            margin-right: 5px;
        }
        .nav-tabs .nav-link.active {
            background: #6f42c1;
            color: white;
            border-color: #6f42c1;
        }
        .password-strength {
            height: 5px;
            border-radius: 3px;
            margin-top: 5px;
            transition: all 0.3s;
        }
        @media (max-width: 768px) {
            .sidebar {
                width: 100%;
                position: relative;
                min-height: auto;
            }
            .main-container {
                margin-left: 0;
            }
        }
        #tablaUsuarios {
            border-radius: 10px;
            overflow: hidden;
        }
        #tablaUsuarios thead {
            background: linear-gradient(to right, var(--celeste-oscuro), #2196f3);
            color: white;
        }
        .user-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--celeste-primario), var(--celeste-secundario));
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: bold;
            color: var(--celeste-oscuro);
            margin-right: 10px;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 col-lg-2 d-md-block sidebar">
                <div class="position-sticky pt-3">
                    <div class="text-center mb-4 p-3">
                        <div class="p-3">
                            <i class="fas fa-building fa-3x mb-3"></i>
                            <h5>Sistema de Visitas</h5>
                            <small>Entidad Pública</small>
                        </div>
                    </div>
                    
                    <div class="nav flex-column">
                        <a class="nav-link" href="../../dashboard.php">
                            <i class="fas fa-home me-2"></i>Dashboard
                        </a>
                        <a class="nav-link" href="../visitas/registrar.php">
                            <i class="fas fa-sign-in-alt me-2"></i>Registrar Ingreso
                        </a>
                        <a class="nav-link" href="../visitas/salida.php">
                            <i class="fas fa-sign-out-alt me-2"></i>Registrar Salida
                        </a>
                        <a class="nav-link" href="../visitas/listar.php">
                            <i class="fas fa-list me-2"></i>Listar Visitas
                        </a>
                        <a class="nav-link" href="../reportes/generar.php">
                            <i class="fas fa-chart-bar me-2"></i>Reportes
                        </a>
                        
                        <hr class="my-3 text-white-50">
                        <h6 class="sidebar-heading px-3 mt-4 mb-1 text-white-50">
                            <span>Administración</span>
                        </h6>
                        <a class="nav-link active" href="gestion.php">
                            <i class="fas fa-users me-2"></i>Gestión de Usuarios
                        </a>
                        <a class="nav-link" href="../areas/gestion.php">
                            <i class="fas fa-sitemap me-2"></i>Áreas
                        </a>
                        <a class="nav-link" href="../entidad/configuracion.php">
                            <i class="fas fa-cog me-2"></i>Configuración
                        </a>
                        
                        <hr class="my-3 text-white-50">
                        <a class="nav-link" href="../../logout.php">
                            <i class="fas fa-sign-out-alt me-2"></i>Cerrar Sesión
                        </a>
                    </div>
                </div>
            </div>

            <!-- Main Content -->
            <div class="col-md-9 ms-sm-auto col-lg-10 px-md-4 main-container">
                <!-- Header -->
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Gestión de Usuarios</h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <span class="badge bg-primary">
                            <i class="fas fa-user-shield me-1"></i>Administrador
                        </span>
                    </div>
                </div>
                
                <?php if ($mensaje): ?>
                <div class="alert alert-<?php echo $tipo_mensaje; ?> alert-dismissible fade show" role="alert">
                    <i class="fas fa-<?php echo $tipo_mensaje == 'success' ? 'check-circle' : 'exclamation-circle'; ?> me-2"></i>
                    <?php echo $mensaje; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
                <?php endif; ?>
                
                <!-- Pestañas -->
                <ul class="nav nav-tabs" id="myTab" role="tablist">
                    <li class="nav-item" role="presentation">
                        <button class="nav-link active" id="lista-tab" data-bs-toggle="tab" 
                                data-bs-target="#lista" type="button" role="tab">
                            <i class="fas fa-list me-2"></i>Lista de Usuarios
                        </button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="nuevo-tab" data-bs-toggle="tab" 
                                data-bs-target="#nuevo" type="button" role="tab">
                            <i class="fas fa-user-plus me-2"></i>
                            <?php echo $usuario_editar ? 'Editar Usuario' : 'Nuevo Usuario'; ?>
                        </button>
                    </li>
                </ul>
                
                <div class="tab-content" id="myTabContent">
                    <!-- Tab Lista de Usuarios -->
                    <div class="tab-pane fade show active" id="lista" role="tabpanel">
                        <div class="table-responsive">
                            <table id="tablaUsuarios" class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Usuario</th>
                                        <th>DNI</th>
                                        <th>Correo</th>
                                        <th>Área</th>
                                        <th>Tipo</th>
                                        <th>Estado</th>
                                        <th>Último Acceso</th>
                                        <th>Acciones</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($usuarios as $usuario): 
                                        // Avatar con iniciales
                                        $iniciales = substr($usuario['nombres'], 0, 1) . substr($usuario['apellidos'], 0, 1);
                                        $iniciales = strtoupper($iniciales);
                                    ?>
                                    <tr>
                                        <td>
                                            <div class="d-flex align-items-center">
                                                <div class="user-avatar">
                                                    <?php echo $iniciales; ?>
                                                </div>
                                                <div>
                                                    <strong><?php echo htmlspecialchars($usuario['apellidos'] . ', ' . $usuario['nombres']); ?></strong>
                                                    <br>
                                                    <small class="text-muted">
                                                        ID: <?php echo $usuario['id']; ?>
                                                    </small>
                                                </div>
                                            </div>
                                        </td>
                                        <td>
                                            <span class="badge bg-secondary">
                                                <?php echo htmlspecialchars($usuario['dni']); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <small><?php echo htmlspecialchars($usuario['correo']); ?></small>
                                            <br>
                                            <?php if ($usuario['email_verificado']): ?>
                                                <span class="badge bg-success badge-sm">✓ Verificado</span>
                                            <?php else: ?>
                                                <span class="badge bg-warning badge-sm">⚠ Pendiente</span>
                                            <?php endif; ?>
                                        </td>
                                        <td><?php echo htmlspecialchars($usuario['area_nombre'] ?? 'No asignada'); ?></td>
                                        <td>
                                            <span class="badge <?php echo $usuario['tipo_usuario'] == 'admin' ? 'badge-admin' : 'badge-registrador'; ?>">
                                                <i class="fas fa-<?php echo $usuario['tipo_usuario'] == 'admin' ? 'user-shield' : 'user'; ?> me-1"></i>
                                                <?php echo ucfirst($usuario['tipo_usuario']); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <span class="badge-estado <?php echo $usuario['activo'] ? 'badge-activo' : 'badge-inactivo'; ?>">
                                                <i class="fas fa-<?php echo $usuario['activo'] ? 'check-circle' : 'times-circle'; ?> me-1"></i>
                                                <?php echo $usuario['activo'] ? 'Activo' : 'Inactivo'; ?>
                                            </span>
                                        </td>
                                        <td>
                                            <small>
                                                <?php 
                                                if ($usuario['fecha_registro']) {
                                                    echo date('d/m/Y', strtotime($usuario['fecha_registro']));
                                                } else {
                                                    echo 'Nunca';
                                                }
                                                ?>
                                            </small>
                                        </td>
                                        <td>
                                            <div class="btn-group btn-group-sm">
                                                <!-- Editar -->
                                                <form method="POST" action="" style="display: inline;">
                                                    <input type="hidden" name="accion" value="cargar_editar">
                                                    <input type="hidden" name="id" value="<?php echo $usuario['id']; ?>">
                                                    <button type="submit" class="btn btn-outline-primary btn-sm" 
                                                            data-bs-toggle="tooltip" title="Editar usuario">
                                                        <i class="fas fa-edit"></i>
                                                    </button>
                                                </form>
                                                
                                                <!-- Cambiar contraseña -->
                                                <button type="button" class="btn btn-outline-warning btn-sm" 
                                                        data-bs-toggle="modal" data-bs-target="#modalPassword"
                                                        data-user-id="<?php echo $usuario['id']; ?>"
                                                        data-user-name="<?php echo htmlspecialchars($usuario['nombres'] . ' ' . $usuario['apellidos']); ?>"
                                                        title="Cambiar contraseña">
                                                    <i class="fas fa-key"></i>
                                                </button>
                                                
                                                <!-- Activar/Desactivar -->
                                                <form method="POST" action="" style="display: inline;">
                                                    <input type="hidden" name="accion" value="toggle_estado">
                                                    <input type="hidden" name="id" value="<?php echo $usuario['id']; ?>">
                                                    <button type="submit" class="btn btn-outline-<?php echo $usuario['activo'] ? 'warning' : 'success'; ?> btn-sm"
                                                            data-bs-toggle="tooltip" 
                                                            title="<?php echo $usuario['activo'] ? 'Desactivar usuario' : 'Activar usuario'; ?>">
                                                        <i class="fas fa-<?php echo $usuario['activo'] ? 'ban' : 'check'; ?>"></i>
                                                    </button>
                                                </form>
                                                
                                                <!-- Eliminar -->
                                                <form method="POST" action="" style="display: inline;" 
                                                      onsubmit="return confirm('¿Está seguro de eliminar este usuario?');">
                                                    <input type="hidden" name="accion" value="eliminar">
                                                    <input type="hidden" name="id" value="<?php echo $usuario['id']; ?>">
                                                    <button type="submit" class="btn btn-outline-danger btn-sm"
                                                            data-bs-toggle="tooltip" title="Eliminar usuario"
                                                            <?php echo ($usuario['id'] == $_SESSION['user_id']) ? 'disabled' : ''; ?>>
                                                        <i class="fas fa-trash"></i>
                                                    </button>
                                                </form>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        
                        <div class="mt-3">
                            <div class="alert alert-info">
                                <i class="fas fa-info-circle me-2"></i>
                                Total de usuarios: <strong><?php echo count($usuarios); ?></strong> | 
                                Administradores: <strong><?php echo count(array_filter($usuarios, fn($u) => $u['tipo_usuario'] == 'admin')); ?></strong> | 
                                Registradores: <strong><?php echo count(array_filter($usuarios, fn($u) => $u['tipo_usuario'] == 'registrador')); ?></strong>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Tab Nuevo/Editar Usuario -->
                    <div class="tab-pane fade" id="nuevo" role="tabpanel">
                        <form method="POST" action="" id="formUsuario">
                            <?php if ($usuario_editar): ?>
                                <input type="hidden" name="accion" value="editar">
                                <input type="hidden" name="id" value="<?php echo $usuario_editar['id']; ?>">
                            <?php else: ?>
                                <input type="hidden" name="accion" value="crear">
                            <?php endif; ?>
                            
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="dni" class="form-label">DNI *</label>
                                    <input type="text" class="form-control" id="dni" name="dni" 
                                           value="<?php echo htmlspecialchars($usuario_editar['dni'] ?? ($_POST['dni'] ?? '')); ?>" 
                                           required placeholder="Ingrese el DNI del usuario">
                                </div>
                                
                                <div class="col-md-6 mb-3">
                                    <label for="correo" class="form-label">Correo Electrónico *</label>
                                    <input type="email" class="form-control" id="correo" name="correo" 
                                           value="<?php echo htmlspecialchars($usuario_editar['correo'] ?? ($_POST['correo'] ?? '')); ?>" 
                                           required placeholder="usuario@entidad.gob.pe">
                                </div>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="apellidos" class="form-label">Apellidos *</label>
                                    <input type="text" class="form-control" id="apellidos" name="apellidos" 
                                           value="<?php echo htmlspecialchars($usuario_editar['apellidos'] ?? ($_POST['apellidos'] ?? '')); ?>" 
                                           required placeholder="Apellidos del usuario">
                                </div>
                                
                                <div class="col-md-6 mb-3">
                                    <label for="nombres" class="form-label">Nombres *</label>
                                    <input type="text" class="form-control" id="nombres" name="nombres" 
                                           value="<?php echo htmlspecialchars($usuario_editar['nombres'] ?? ($_POST['nombres'] ?? '')); ?>" 
                                           required placeholder="Nombres del usuario">
                                </div>
                            </div>
                            
                            <?php if (!$usuario_editar): ?>
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="password" class="form-label">Contraseña *</label>
                                    <input type="password" class="form-control" id="password" name="password" 
                                           required minlength="6" placeholder="Mínimo 6 caracteres">
                                    <div class="password-strength" id="passwordStrength"></div>
                                    <small class="text-muted" id="passwordHint"></small>
                                </div>
                                
                                <div class="col-md-6 mb-3">
                                    <label for="confirm_password" class="form-label">Confirmar Contraseña *</label>
                                    <input type="password" class="form-control" id="confirm_password" 
                                           name="confirm_password" required minlength="6" placeholder="Repita la contraseña">
                                    <small class="text-muted" id="passwordMatch"></small>
                                </div>
                            </div>
                            <?php endif; ?>
                            
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="area_id" class="form-label">Área *</label>
                                    <select class="form-select" id="area_id" name="area_id" required>
                                        <option value="">Seleccione un área</option>
                                        <?php foreach ($areas as $area): ?>
                                            <option value="<?php echo $area['id']; ?>" 
                                                <?php echo (($usuario_editar['area_id'] ?? ($_POST['area_id'] ?? '')) == $area['id']) ? 'selected' : ''; ?>>
                                                <?php echo htmlspecialchars($area['nombre_area']); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                
                                <div class="col-md-6 mb-3">
                                    <label for="tipo_usuario" class="form-label">Tipo de Usuario *</label>
                                    <select class="form-select" id="tipo_usuario" name="tipo_usuario" required>
                                        <option value="registrador" <?php echo (($usuario_editar['tipo_usuario'] ?? ($_POST['tipo_usuario'] ?? '')) == 'registrador') ? 'selected' : ''; ?>>Registrador</option>
                                        <option value="admin" <?php echo (($usuario_editar['tipo_usuario'] ?? ($_POST['tipo_usuario'] ?? '')) == 'admin') ? 'selected' : ''; ?>>Administrador</option>
                                    </select>
                                </div>
                            </div>
                            
                            <div class="d-flex justify-content-between mt-4">
                                <?php if ($usuario_editar): ?>
                                    <a href="gestion.php" class="btn btn-secondary">
                                        <i class="fas fa-times me-1"></i> Cancelar Edición
                                    </a>
                                <?php else: ?>
                                    <button type="reset" class="btn btn-secondary">
                                        <i class="fas fa-redo me-1"></i> Limpiar Formulario
                                    </button>
                                <?php endif; ?>
                                
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-save me-1"></i>
                                    <?php echo $usuario_editar ? 'Actualizar Usuario' : 'Crear Usuario'; ?>
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Modal para cambiar contraseña -->
    <div class="modal fade" id="modalPassword" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title"><i class="fas fa-key me-2"></i>Cambiar Contraseña</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST" action="" id="formPassword">
                    <div class="modal-body">
                        <input type="hidden" name="accion" value="cambiar_password">
                        <input type="hidden" name="id" id="passwordUserId">
                        
                        <p>Usuario: <strong id="passwordUserName"></strong></p>
                        
                        <div class="mb-3">
                            <label for="modal_password" class="form-label">Nueva Contraseña *</label>
                            <input type="password" class="form-control" id="modal_password" name="password" 
                                   required minlength="6" placeholder="Mínimo 6 caracteres">
                        </div>
                        
                        <div class="mb-3">
                            <label for="modal_confirm_password" class="form-label">Confirmar Contraseña *</label>
                            <input type="password" class="form-control" id="modal_confirm_password" 
                                   name="confirm_password" required minlength="6" placeholder="Repita la contraseña">
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="submit" class="btn btn-primary">Cambiar Contraseña</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.datatables.net/1.11.5/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.11.5/js/dataTables.bootstrap5.min.js"></script>
    
    <script>
        // Inicializar DataTable
        $(document).ready(function() {
            $('#tablaUsuarios').DataTable({
                language: {
                    url: '//cdn.datatables.net/plug-ins/1.11.5/i18n/es-ES.json'
                },
                pageLength: 25,
                order: [[0, 'asc']],
                dom: '<"row"<"col-md-6"l><"col-md-6"f>>rt<"row"<"col-md-6"i><"col-md-6"p>>',
                columnDefs: [
                    { orderable: false, targets: [7] } // Columna de acciones no ordenable
                ]
            });
            
            // Inicializar tooltips
            var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
            var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
                return new bootstrap.Tooltip(tooltipTriggerEl);
            });
            
            // Cambiar a pestaña de edición si hay usuario para editar
            <?php if ($usuario_editar): ?>
            var nuevoTab = new bootstrap.Tab(document.getElementById('nuevo-tab'));
            nuevoTab.show();
            <?php endif; ?>
        });
        
        // Configurar modal de cambiar contraseña
        document.getElementById('modalPassword').addEventListener('show.bs.modal', function(event) {
            var button = event.relatedTarget;
            var userId = button.getAttribute('data-user-id');
            var userName = button.getAttribute('data-user-name');
            
            document.getElementById('passwordUserId').value = userId;
            document.getElementById('passwordUserName').textContent = userName;
        });
        
        // Validar formulario de usuario
        document.getElementById('formUsuario').addEventListener('submit', function(e) {
            <?php if (!$usuario_editar): ?>
            // Validar contraseñas solo para creación
            var password = document.getElementById('password').value;
            var confirmPassword = document.getElementById('confirm_password').value;
            
            if (password.length < 6) {
                e.preventDefault();
                alert('La contraseña debe tener al menos 6 caracteres');
                return false;
            }
            
            if (password !== confirmPassword) {
                e.preventDefault();
                alert('Las contraseñas no coinciden');
                return false;
            }
            <?php endif; ?>
            
            // Validar DNI (8 dígitos para Perú)
            var dni = document.getElementById('dni').value.trim();
            if (!/^\d{8}$/.test(dni)) {
                e.preventDefault();
                alert('El DNI debe tener 8 dígitos numéricos');
                return false;
            }
            
            // Validar email
            var email = document.getElementById('correo').value.trim();
            var emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            if (!emailRegex.test(email)) {
                e.preventDefault();
                alert('Por favor ingrese un correo electrónico válido');
                return false;
            }
            
            return true;
        });
        
        // Validar formulario de cambiar contraseña
        document.getElementById('formPassword').addEventListener('submit', function(e) {
            var password = document.getElementById('modal_password').value;
            var confirmPassword = document.getElementById('modal_confirm_password').value;
            
            if (password.length < 6) {
                e.preventDefault();
                alert('La contraseña debe tener al menos 6 caracteres');
                return false;
            }
            
            if (password !== confirmPassword) {
                e.preventDefault();
                alert('Las contraseñas no coinciden');
                return false;
            }
            
            return true;
        });
        
        // Mostrar fortaleza de contraseña (solo para creación)
        <?php if (!$usuario_editar): ?>
        document.getElementById('password').addEventListener('input', function() {
            var password = this.value;
            var strengthBar = document.getElementById('passwordStrength');
            var hint = document.getElementById('passwordHint');
            
            let strength = 0;
            let color = '#dc3545';
            let message = 'Muy débil';
            
            // Verificar longitud
            if (password.length >= 6) strength++;
            if (password.length >= 8) strength++;
            
            // Verificar caracteres diversos
            if (/[a-z]/.test(password)) strength++;
            if (/[A-Z]/.test(password)) strength++;
            if (/[0-9]/.test(password)) strength++;
            if (/[^A-Za-z0-9]/.test(password)) strength++;
            
            // Determinar color y mensaje
            switch (strength) {
                case 0:
                case 1:
                    color = '#dc3545';
                    message = 'Muy débil';
                    break;
                case 2:
                    color = '#fd7e14';
                    message = 'Débil';
                    break;
                case 3:
                    color = '#ffc107';
                    message = 'Regular';
                    break;
                case 4:
                    color = '#20c997';
                    message = 'Buena';
                    break;
                case 5:
                case 6:
                    color = '#28a745';
                    message = 'Excelente';
                    break;
            }
            
            // Actualizar barra y mensaje
            strengthBar.style.width = (strength / 6 * 100) + '%';
            strengthBar.style.backgroundColor = color;
            hint.textContent = message;
            hint.style.color = color;
        });
        
        // Verificar que las contraseñas coincidan
        document.getElementById('confirm_password').addEventListener('input', function() {
            var password = document.getElementById('password').value;
            var confirmPassword = this.value;
            var matchText = document.getElementById('passwordMatch');
            
            if (confirmPassword === '') {
                matchText.textContent = '';
                matchText.style.color = '';
            } else if (password === confirmPassword) {
                matchText.textContent = '✓ Las contraseñas coinciden';
                matchText.style.color = '#28a745';
            } else {
                matchText.textContent = '✗ Las contraseñas no coinciden';
                matchText.style.color = '#dc3545';
            }
        });
        <?php endif; ?>
    </script>
</body>
</html>