<?php
// Habilitar errores temporalmente
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

session_start();

// Verificar sesión
if (!isset($_SESSION['user_id'])) {
    header("Location: ../../login.php");
    exit();
}

require_once '../../config/database.php';

$database = new Database();
$db = $database->getConnection();

if (!$db) {
    die("Error de conexión a la base de datos");
}

// Obtener áreas para el select
$areas = [];
try {
    $query = "SELECT id, nombre_area FROM areas WHERE activo = 1 ORDER BY nombre_area";
    $stmt = $db->prepare($query);
    $stmt->execute();
    $areas = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    error_log("Error al cargar áreas: " . $e->getMessage());
}

$mensaje = '';
$tipo_mensaje = '';

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // Sanitizar y validar datos
    $documento = trim($_POST['documento'] ?? '');
    $apellidos = trim($_POST['apellidos'] ?? '');
    $nombres = trim($_POST['nombres'] ?? '');
    $area_id = $_POST['area_id'] ?? 0;
    $motivo = trim($_POST['motivo'] ?? '');
    $tipo_ingreso = $_POST['tipo_ingreso'] ?? 'automatico';
    
    // Validaciones básicas
    if (empty($documento) || empty($apellidos) || empty($nombres) || empty($area_id) || empty($motivo)) {
        $mensaje = "Todos los campos son obligatorios.";
        $tipo_mensaje = "danger";
    } else {
        // Determinar hora de ingreso
        if ($tipo_ingreso == 'manual' && isset($_POST['hora_ingreso'])) {
            $hora_ingreso = $_POST['hora_ingreso'];
        } else {
            $hora_ingreso = date('H:i:s');
        }
        
        try {
            $query = "INSERT INTO visitas (
                fecha_visita, 
                documento_visitante, 
                apellidos_visitante, 
                nombres_visitante, 
                area_id, 
                hora_ingreso, 
                motivo_visita, 
                usuario_registra_id, 
                tipo_ingreso
            ) VALUES (
                CURDATE(), 
                :documento, 
                :apellidos, 
                :nombres, 
                :area_id, 
                :hora_ingreso, 
                :motivo, 
                :usuario_id, 
                :tipo_ingreso
            )";
            
            $stmt = $db->prepare($query);
            $stmt->bindParam(':documento', $documento);
            $stmt->bindParam(':apellidos', $apellidos);
            $stmt->bindParam(':nombres', $nombres);
            $stmt->bindParam(':area_id', $area_id, PDO::PARAM_INT);
            $stmt->bindParam(':hora_ingreso', $hora_ingreso);
            $stmt->bindParam(':motivo', $motivo);
            $stmt->bindParam(':usuario_id', $_SESSION['user_id'], PDO::PARAM_INT);
            $stmt->bindParam(':tipo_ingreso', $tipo_ingreso);
            
            if ($stmt->execute()) {
                $mensaje = "Visita registrada exitosamente!";
                $tipo_mensaje = "success";
                
                // Limpiar campos del formulario (opcional)
                $_POST = array();
            } else {
                $mensaje = "Error al registrar la visita.";
                $tipo_mensaje = "danger";
            }
        } catch (PDOException $e) {
            $mensaje = "Error: " . $e->getMessage();
            $tipo_mensaje = "danger";
            error_log("Error en registro de visita: " . $e->getMessage());
        }
    }
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Registrar Visita - Sistema de Visitas</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        :root {
            --celeste-primario: #e3f2fd;
            --celeste-secundario: #bbdefb;
            --celeste-oscuro: #1976d2;
        }
        body {
            background-color: #f8f9fa;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        .main-container {
            margin-left: 250px;
            padding: 20px;
            min-height: 100vh;
        }
        .sidebar {
            background: linear-gradient(180deg, var(--celeste-oscuro) 0%, #1565c0 100%);
            min-height: 100vh;
            color: white;
            padding: 0;
            position: fixed;
            width: 250px;
        }
        .sidebar .nav-link {
            color: rgba(255,255,255,0.8);
            padding: 12px 20px;
            margin: 2px 10px;
            border-radius: 8px;
            transition: all 0.3s;
            text-decoration: none;
        }
        .sidebar .nav-link:hover, .sidebar .nav-link.active {
            background: rgba(255,255,255,0.1);
            color: white;
            transform: translateX(5px);
        }
        .card {
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.05);
            border: none;
            margin-bottom: 20px;
        }
        .card-header {
            background: linear-gradient(to right, var(--celeste-oscuro), #2196f3);
            color: white;
            border-radius: 15px 15px 0 0 !important;
            padding: 15px 20px;
            font-weight: 600;
        }
        .btn-primary {
            background: linear-gradient(to right, var(--celeste-oscuro), #2196f3);
            border: none;
            padding: 10px 25px;
            border-radius: 8px;
            transition: all 0.3s;
        }
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(33, 150, 243, 0.4);
        }
        .btn-success {
            background: linear-gradient(to right, #28a745, #20c997);
            border: none;
        }
        .form-control, .form-select {
            border-radius: 8px;
            border: 2px solid #e0e0e0;
            padding: 10px 15px;
            transition: all 0.3s;
        }
        .form-control:focus, .form-select:focus {
            border-color: var(--celeste-oscuro);
            box-shadow: 0 0 0 0.2rem rgba(25, 118, 210, 0.25);
        }
        @media (max-width: 768px) {
            .sidebar {
                width: 100%;
                position: relative;
                min-height: auto;
            }
            .main-container {
                margin-left: 0;
            }
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 col-lg-2 d-md-block sidebar">
                <div class="position-sticky pt-3">
                    <div class="text-center mb-4 p-3">
                        <div class="p-3">
                            <i class="fas fa-building fa-3x mb-3"></i>
                            <h5>Sistema de Visitas</h5>
                            <small>Entidad Pública</small>
                        </div>
                    </div>
                    
                    <div class="nav flex-column">
                        <a class="nav-link" href="../../dashboard.php">
                            <i class="fas fa-home me-2"></i>Dashboard
                        </a>
                        <a class="nav-link active" href="registrar.php">
                            <i class="fas fa-sign-in-alt me-2"></i>Registrar Ingreso
                        </a>
                        <a class="nav-link" href="salida.php">
                            <i class="fas fa-sign-out-alt me-2"></i>Registrar Salida
                        </a>
                        <a class="nav-link" href="listar.php">
                            <i class="fas fa-list me-2"></i>Listar Visitas
                        </a>
                        <a class="nav-link" href="../reportes/generar.php">
                            <i class="fas fa-chart-bar me-2"></i>Reportes
                        </a>
                        
                        <?php if (isset($_SESSION['user_tipo']) && $_SESSION['user_tipo'] == 'admin'): ?>
                        <hr class="my-3 text-white-50">
                        <h6 class="sidebar-heading px-3 mt-4 mb-1 text-white-50">
                            <span>Administración</span>
                        </h6>
                        <a class="nav-link" href="../usuarios/gestion.php">
                            <i class="fas fa-users me-2"></i>Gestión de Usuarios
                        </a>
                        <a class="nav-link" href="../areas/gestion.php">
                            <i class="fas fa-sitemap me-2"></i>Áreas
                        </a>
                        <a class="nav-link" href="../entidad/configuracion.php">
                            <i class="fas fa-cog me-2"></i>Configuración
                        </a>
                        <?php endif; ?>
                        
                        <hr class="my-3 text-white-50">
                        <a class="nav-link" href="../../logout.php">
                            <i class="fas fa-sign-out-alt me-2"></i>Cerrar Sesión
                        </a>
                    </div>
                </div>
            </div>

            <!-- Main Content -->
            <div class="col-md-9 ms-sm-auto col-lg-10 px-md-4 main-container">
                <!-- Header -->
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Registrar Nueva Visita</h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <span class="badge bg-primary">
                            <i class="fas fa-user me-1"></i>
                            <?php echo isset($_SESSION['user_nombre']) ? htmlspecialchars($_SESSION['user_nombre']) : 'Usuario'; ?>
                        </span>
                    </div>
                </div>
                
                <?php if ($mensaje): ?>
                <div class="alert alert-<?php echo $tipo_mensaje; ?> alert-dismissible fade show" role="alert">
                    <i class="fas fa-<?php echo $tipo_mensaje == 'success' ? 'check-circle' : 'exclamation-circle'; ?> me-2"></i>
                    <?php echo $mensaje; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
                <?php endif; ?>
                
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="fas fa-user-plus me-2"></i>Datos del Visitante</h5>
                    </div>
                    <div class="card-body">
                        <form method="POST" action="" id="formVisita">
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="documento" class="form-label">
                                        <i class="fas fa-id-card me-1"></i>Documento de Identidad (DNI)
                                    </label>
                                    <div class="input-group">
                                        <input type="text" class="form-control" id="documento" name="documento" 
                                               required placeholder="Ingrese DNI del visitante"
                                               value="<?php echo isset($_POST['documento']) ? htmlspecialchars($_POST['documento']) : ''; ?>">
                                        <button type="button" class="btn btn-outline-primary" id="btnBuscarDNI">
                                            <i class="fas fa-search me-1"></i> Buscar
                                        </button>
                                    </div>
                                    <small class="text-muted">Presione Enter o el botón Buscar para consultar automáticamente</small>
                                </div>
                                
                                <div class="col-md-6 mb-3">
                                    <label for="area_id" class="form-label">
                                        <i class="fas fa-sitemap me-1"></i>Área a Visitar
                                    </label>
                                    <select class="form-select" id="area_id" name="area_id" required>
                                        <option value="">Seleccione un área</option>
                                        <?php foreach ($areas as $area): ?>
                                            <option value="<?php echo $area['id']; ?>" 
                                                <?php echo (isset($_POST['area_id']) && $_POST['area_id'] == $area['id']) ? 'selected' : ''; ?>>
                                                <?php echo htmlspecialchars($area['nombre_area']); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="apellidos" class="form-label">
                                        <i class="fas fa-user me-1"></i>Apellidos
                                    </label>
                                    <input type="text" class="form-control" id="apellidos" name="apellidos" 
                                           required placeholder="Apellidos del visitante"
                                           value="<?php echo isset($_POST['apellidos']) ? htmlspecialchars($_POST['apellidos']) : ''; ?>">
                                </div>
                                
                                <div class="col-md-6 mb-3">
                                    <label for="nombres" class="form-label">
                                        <i class="fas fa-user me-1"></i>Nombres
                                    </label>
                                    <input type="text" class="form-control" id="nombres" name="nombres" 
                                           required placeholder="Nombres del visitante"
                                           value="<?php echo isset($_POST['nombres']) ? htmlspecialchars($_POST['nombres']) : ''; ?>">
                                </div>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="tipo_ingreso" class="form-label">
                                        <i class="fas fa-clock me-1"></i>Tipo de Registro de Hora
                                    </label>
                                    <select class="form-select" id="tipo_ingreso" name="tipo_ingreso" required>
                                        <option value="automatico" <?php echo (!isset($_POST['tipo_ingreso']) || $_POST['tipo_ingreso'] == 'automatico') ? 'selected' : ''; ?>>
                                            Automático (Hora del servidor)
                                        </option>
                                        <option value="manual" <?php echo (isset($_POST['tipo_ingreso']) && $_POST['tipo_ingreso'] == 'manual') ? 'selected' : ''; ?>>
                                            Manual (Ingresar hora)
                                        </option>
                                    </select>
                                </div>
                                
                                <div class="col-md-6 mb-3" id="hora_manual_container" 
                                     style="display: <?php echo (isset($_POST['tipo_ingreso']) && $_POST['tipo_ingreso'] == 'manual') ? 'block' : 'none'; ?>;">
                                    <label for="hora_ingreso" class="form-label">
                                        <i class="fas fa-clock me-1"></i>Hora de Ingreso
                                    </label>
                                    <input type="time" class="form-control" id="hora_ingreso" name="hora_ingreso" 
                                           value="<?php echo isset($_POST['hora_ingreso']) ? $_POST['hora_ingreso'] : date('H:i'); ?>">
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="motivo" class="form-label">
                                    <i class="fas fa-comment me-1"></i>Motivo de la Visita
                                </label>
                                <textarea class="form-control" id="motivo" name="motivo" rows="3" 
                                          required placeholder="Describa el motivo de la visita"><?php echo isset($_POST['motivo']) ? htmlspecialchars($_POST['motivo']) : ''; ?></textarea>
                            </div>
                            
                            <div class="d-flex justify-content-between mt-4">
                                <a href="../../dashboard.php" class="btn btn-secondary">
                                    <i class="fas fa-arrow-left me-1"></i> Volver al Dashboard
                                </a>
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-save me-1"></i> Registrar Visita
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Mostrar/ocultar campo de hora manual
        document.getElementById('tipo_ingreso').addEventListener('change', function() {
            var container = document.getElementById('hora_manual_container');
            if (this.value === 'manual') {
                container.style.display = 'block';
            } else {
                container.style.display = 'none';
            }
        });
        
        // Consultar DNI (simulación)
        document.getElementById('btnBuscarDNI').addEventListener('click', buscarDNI);
        document.getElementById('documento').addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                e.preventDefault();
                buscarDNI();
            }
        });
        
        function buscarDNI() {
            var dni = document.getElementById('documento').value.trim();
            if (dni.length === 0) {
                mostrarMensaje('warning', 'Por favor ingrese un número de DNI');
                return;
            }
            
            if (dni.length !== 8) {
                mostrarMensaje('warning', 'El DNI debe tener 8 dígitos');
                return;
            }
            
            // Mostrar loading
            var btn = document.getElementById('btnBuscarDNI');
            var originalHTML = btn.innerHTML;
            btn.innerHTML = '<i class="fas fa-spinner fa-spin me-1"></i> Buscando...';
            btn.disabled = true;
            
            // Simulación de búsqueda (en producción usar API real)
            setTimeout(function() {
                // Datos de prueba
                var nombresEjemplo = ['Juan', 'Maria', 'Carlos', 'Ana', 'Luis'];
                var apellidosEjemplo = ['Perez', 'Gomez', 'Rodriguez', 'Lopez', 'Martinez'];
                
                var nombreAleatorio = nombresEjemplo[Math.floor(Math.random() * nombresEjemplo.length)];
                var apellidoAleatorio = apellidosEjemplo[Math.floor(Math.random() * apellidosEjemplo.length)];
                
                // Rellenar campos
                document.getElementById('apellidos').value = apellidoAleatorio;
                document.getElementById('nombres').value = nombreAleatorio;
                
                mostrarMensaje('success', 'Datos encontrados: ' + apellidoAleatorio + ', ' + nombreAleatorio);
                
                btn.innerHTML = originalHTML;
                btn.disabled = false;
            }, 1500);
        }
        
        function mostrarMensaje(tipo, mensaje) {
            // Remover mensajes anteriores
            var mensajesAnteriores = document.querySelectorAll('.mensaje-temporal');
            mensajesAnteriores.forEach(function(msg) {
                msg.remove();
            });
            
            var alertDiv = document.createElement('div');
            alertDiv.className = 'alert alert-' + tipo + ' alert-dismissible fade show mensaje-temporal mt-2';
            alertDiv.innerHTML = '<i class="fas fa-' + (tipo === 'success' ? 'check-circle' : 'exclamation-circle') + ' me-2"></i>' + 
                                mensaje + 
                                '<button type="button" class="btn-close" data-bs-dismiss="alert"></button>';
            
            document.getElementById('formVisita').prepend(alertDiv);
            
            // Auto-remover después de 5 segundos
            setTimeout(function() {
                if (alertDiv.parentNode) {
                    alertDiv.classList.remove('show');
                    setTimeout(function() {
                        if (alertDiv.parentNode) {
                            alertDiv.remove();
                        }
                    }, 300);
                }
            }, 5000);
        }
        
        // Validar formulario
        document.getElementById('formVisita').addEventListener('submit', function(e) {
            var documento = document.getElementById('documento').value.trim();
            var apellidos = document.getElementById('apellidos').value.trim();
            var nombres = document.getElementById('nombres').value.trim();
            var area = document.getElementById('area_id').value;
            var motivo = document.getElementById('motivo').value.trim();
            
            if (documento === '' || apellidos === '' || nombres === '' || area === '' || motivo === '') {
                e.preventDefault();
                mostrarMensaje('warning', 'Por favor complete todos los campos obligatorios');
                return false;
            }
            
            return true;
        });
    </script>
</body>
</html>